;;;=======================[ BreakObjects.lsp ]==============================
;;; Author: Copyright 2006-2012 Charles Alan Butler 
;;; Contact @  www.TheSwamp.org    
;;;   http://www.theswamp.org/index.php?topic=10370.0
;;; Version:  2.2  July 28, 2012
;;; Purpose: Break All selected objects
;;;    permitted objects are lines, lwplines, plines, splines,
;;;    ellipse, circles & arcs 
;;;                            
;;;  Function  c:MyBreak -       DCL for selecting the routines
;;;  Function  c:BreakAll -      Break all objects selected with each other
;;;  Function  c:BreakwObject  - Break many objects with a single object
;;;  Function  c:BreakObject -   Break a single object with other objects 
;;;  Function  c:BreakWith -     Break selected objects with other selected objects
;;;  Function  c:BreakTouching - Break objects touching selected objects
;;;  Function  c:BreakSelected - Break selected objects with any objects that touch it 
;;;  Function  c:BreakRemove - Break selected object with any objects that touch it & remove 
;;;                         every other new segment, start with selected object
;;;  Revision 1.8 Added Option for Break Gap greater than zero
;;;  NEW r1.9  c:BreakWlayer -   Break objects with objects on a layer
;;;  NEW r1.9  c:BreakWithTouching - Break touching objects with selected objects
;;;  Revision 2.0 Fixed a bug when point to break is at the end of object
;;;  Revision 2.1 Fixed another bug when point to break is at the end of object
;;;  Revision 2.2 Fixed another bug when closed objects are to be broken
;;;
;;;
;;;  Function  break_with -  main break function called by all others and
;;;                          returns a list of new enames, see c:BreakAll
;;;                          for an example of using the return list
;;;
;;; Requirements: objects must have the same z-value
;;; Restrictions: Does not Break objects on locked layers 
;;; Returns:  none
;;;
;;;=====================================================================
;;;   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED     ;
;;;   WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR  ;
;;;   PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.            ;
;;;                                                                    ;
;;;  You are hereby granted permission to use, copy and modify this    ;
;;;  software without charge, provided you do so exclusively for       ;
;;;  your own use or for use by others in your organization in the     ;
;;;  performance of their normal duties, and provided further that     ;
;;;  the above copyright notice appears in all copies and both that    ;
;;;  copyright notice and the limited warranty and restricted rights   ;
;;;  notice below appear in all supporting documentation.              ;
;;;=====================================================================


;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;;               M A I N   S U B R O U T I N E                   
;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

(defun break_with ( ss2brk ss2brkwith self Gap / cmd intpts lst masterlist ss ssobjs
                    _vl-sort-i onlockedlayer ssget->vla-list list->3pair GetNewEntities oc
                    get_interpts break_obj LastEntInDatabase ss2brkwithList
                    *brkcnt*
                 )
  ;; ss2brk     selection set to break
  ;; ss2brkwith selection set to use as break points
  ;; self       when true will allow an object to break itself
  ;;            note that plined will break at each vertex
  ;;
  ;; return list of enames of new objects
  
  (or (not (vl-catch-all-error-p (vl-catch-all-apply (function vlax-get-acad-object) nil))) (vl-load-com)) 
  
  (princ "\nCalculating Break Points, Please Wait.\n")

;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;;                S U B   F U N C T I O N S                      
;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

  (defun _vl-sort-i ( lst func )
    (mapcar
      (function (lambda ( x ) (nth x lst)))
      (vl-sort-i lst func)
    )
  )

  ;;  return T if entity is on a locked layer
  (defun onlockedlayer (ename / entlst)
    (setq entlst (tblsearch "LAYER" (cdr (assoc 8 (entget ename)))))
    (= 4 (logand 4 (cdr (assoc 70 entlst))))
  )

  ;;  return a list of objects from a selection set
  (defun ssget->vla-list (ss / i ename allobj) ; this is faster, changed in ver 1.7
    (setq i -1)
    (while (setq  ename (ssname ss (setq i (1+ i))))
      (setq allobj (cons (vlax-ename->vla-object ename) allobj))
    )
    allobj
  )
  
  ;;  return a list of lists grouped by 3 from a flat list
  (defun list->3pair (old / new)
    (while (setq new (cons (list (car old) (cadr old) (caddr old)) new)
               old (cdddr old))
    )
    (reverse new)
  )
  
;;=====================================
;;  return a list of intersect points  
;;=====================================

(defun get_interpts (obj1 obj2 / iplist)
  (if (not (vl-catch-all-error-p
             (setq iplist (vl-catch-all-apply
                            'vlax-safearray->list
                            (list
                              (vlax-variant-value
                                (vla-intersectwith obj1 obj2 acextendnone)
                              ))))))
    iplist
  )
)

;;========================================
;;  Break entity at break points in list  
;;========================================
;;   New as per version 1.8 [BrkGap] --- This subroutine has been re-written
;;  Loop through the break points breaking the entity
;;  If the entity is not a closed entity then a new object is created
;;  This object is added to a list. When break points don't fall on the current 
;;  entity the list of new entities are searched to locate the entity that the 
;;  point is on so it can be broken.
;;  "Break with a Gap" has been added to this routine. The problem faced with 
;;  this method is that sections to be removed may lap if the break points are
;;  too close to each other. The solution is to create a list of break point pairs 
;;  representing the gap to be removed and test to see if there i an overlap. If
;;  there is then merge the break point pairs into one large gap. This way the 
;;  points will always fall on an object with one exception. If the gap is too near
;;  the end of an object one break point will be off the end and therefore that 
;;  point will need to be replaced with the end point.
;;    NOTE: in ACAD2000 the (vlax-curve-getdistatpoint function has proven unreliable
;;  so I have used (vlax-curve-getdistatparam in most cases

(defun break_obj ( ent brkptlst BrkGap / _vl-sort-i brkobjlst en enttype maxparam closedobj
                   minparam obj obj2break p1param p2param brkpt2 dlst idx brkptS
                   brkptE brkpt result GapFlg result ignore dist tmppt
                   #ofpts 2gap enddist lastent obj2break stdist spt ept
                 )
  (or BrkGap (setq BrkGap 0.0)) ; default to 0
  (setq BrkGap (/ BrkGap 2.0)) ; if Gap use 1/2 per side of break point
  
  (setq obj2break ent
        brkobjlst (list ent)
        enttype   (cdr (assoc 0 (entget ent)))
        GapFlg    (not (zerop BrkGap)) ; gap > 0
        closedobj (vlax-curve-isclosed obj2break)
  )
  
  (defun _vl-sort-i ( lst func )
    (mapcar
      (function (lambda ( x ) (nth x lst)))
      (vl-sort-i lst func)
    )
  )
  
  ;; when zero gap no need to break at end points, not closed
  (if (and (zerop Brkgap)(not closedobj)) ; Revision 2.2
    (setq spt (vlax-curve-getstartpoint ent)
          ept (vlax-curve-getendpoint ent)
          brkptlst (vl-remove-if '(lambda(x) (or (< (distance x spt) 0.0001)
                                                 (< (distance x ept) 0.0001)))
                                 brkptlst)
    )
  )
  (if brkptlst
    (progn
  ;;  sort break points based on the distance along the break object
  ;;  get distance to break point, catch error if pt is off end
  ;; ver 2.0 fix - added COND to fix break point is at the end of a
  ;; line which is not a valid break but does no harm
  (setq brkptlst (mapcar '(lambda(x) (list x (vlax-curve-getdistatparam obj2break
                                               ;; ver 2.0 fix
                                               (cond ((vlax-curve-getparamatpoint obj2break x))
                                                   ((vlax-curve-getparamatpoint obj2break
                                                     (vlax-curve-getclosestpointto obj2break x))))))
                            ) brkptlst))
  ;; sort primary list on distance
  (setq brkptlst (_vl-sort-i brkptlst '(lambda (a1 a2) (< (cadr a1) (cadr a2)))))
  
  (if GapFlg ; gap > 0
    ;; Brkptlst starts as the break point and then a list of pairs of points
    ;;  is creates as the break points
    (progn
      ;;  create a list of list of break points
      ;;  ((idx# stpoint distance)(idx# endpoint distance)...)
      (setq idx 0)
      (foreach brkpt brkptlst
        
        ;; ----------------------------------------------------------
        ;;  create start break point, then create end break point    
        ;;  ((idx# startpoint distance)(idx# endpoint distance)...)  
        ;; ----------------------------------------------------------
        (setq dist (cadr brkpt)) ; distance to center of gap
        ;;  subtract gap to get start point of break gap
        (cond
          ((and (minusp (setq stDist (- dist BrkGap))) closedobj )
           (setq stdist (+ (vlax-curve-getdistatparam obj2break
                             (vlax-curve-getendparam obj2break)) stDist))
           (setq dlst (cons (list idx
                                  (vlax-curve-getpointatparam obj2break
                                         (vlax-curve-getparamatdist obj2break stDist))
                                  stDist) dlst))
           )
          ((minusp stDist) ; off start of object so get startpoint
           (setq dlst (cons (list idx (vlax-curve-getstartpoint obj2break) 0.0) dlst))
           )
          (t
           (setq dlst (cons (list idx
                                  (vlax-curve-getpointatparam obj2break
                                         (vlax-curve-getparamatdist obj2break stDist))
                                  stDist) dlst))
          )
        )
        ;;  add gap to get end point of break gap
        (cond
          ((and (> (setq stDist (+ dist BrkGap))
                   (setq endDist (vlax-curve-getdistatparam obj2break
                                     (vlax-curve-getendparam obj2break)))) closedobj )
           (setq stdist (- stDist endDist))
           (setq dlst (cons (list idx
                                  (vlax-curve-getpointatparam obj2break
                                         (vlax-curve-getparamatdist obj2break stDist))
                                  stDist) dlst))
           )
          ((> stDist endDist) ; off end of object so get endpoint
           (setq dlst (cons (list idx
                                  (vlax-curve-getpointatparam obj2break
                                        (vlax-curve-getendparam obj2break))
                                  endDist) dlst))
           )
          (t
           (setq dlst (cons (list idx
                                  (vlax-curve-getpointatparam obj2break
                                         (vlax-curve-getparamatdist obj2break stDist))
                                  stDist) dlst))
          )
        )
        ;; -------------------------------------------------------
        (setq idx (1+ IDX))
      ) ; foreach brkpt brkptlst
      

      (setq dlst (reverse dlst))
      ;;  remove the points of the gap segments that overlap
      (setq idx -1
            2gap (* BrkGap 2)
            #ofPts (length Brkptlst)
      )
      (while (<= (setq idx (1+ idx)) #ofPts)
        (cond
          ((null result) ; 1st time through
           (setq result (list (car dlst)) ; get first start point
                 result (cons (nth (1+(* idx 2)) dlst) result))
          )
          ((= idx #ofPts) ; last pass, check for wrap
           (if (and closedobj (> #ofPts 1)
                    (<= (+(- (vlax-curve-getdistatparam obj2break
                            (vlax-curve-getendparam obj2break))
                          (cadr (last BrkPtLst))) (cadar BrkPtLst)) 2Gap))
             (progn
               (if (zerop (rem (length result) 2))
                 (setq result (cdr result)) ; remove the last end point
               )
               ;;  ignore previous endpoint and present start point
               (setq result (cons (cadr (reverse result)) result) ; get last end point
                     result (cdr (reverse result))
                     result (reverse (cdr result)))
             )
           )
          )
          ;; Break Gap Overlaps
          ((< (cadr (nth idx Brkptlst)) (+ (cadr (nth (1- idx) Brkptlst)) 2Gap))
           (if (zerop (rem (length result) 2))
             (setq result (cdr result)) ; remove the last end point
           )
           ;;  ignore previous endpoint and present start point
           (setq result (cons (nth (1+(* idx 2)) dlst) result)) ; get present end point
           )
          ;; Break Gap does Not Overlap previous point 
          (t
           (setq result (cons (nth (* idx 2) dlst) result)) ; get this start point
           (setq result (cons (nth (1+(* idx 2)) dlst) result)) ; get this end point
          )
        ) ; end cond stmt
      ) ; while
      
      ;;  setup brkptlst with pair of break pts ((p1 p2)(p3 p4)...)
      ;;  one of the pair of points will be on the object that
      ;;  needs to be broken
      (setq dlst     (reverse result)
            brkptlst nil)
      (while dlst ; grab the points only
        (setq brkptlst (cons (list (cadar dlst)(cadadr dlst)) brkptlst)
              dlst   (cddr dlst))
      )
    )
  )
  ;;   -----------------------------------------------------

  ;; (if (equal  a ent) (princ)) ; debug CAB  -------------
 
  (foreach brkpt (reverse brkptlst)
    (if GapFlg ; gap > 0
      (setq brkptS (car brkpt)
            brkptE (cadr brkpt))
      (setq brkptS (car brkpt)
            brkptE brkptS)
    )
    ;;  get last entity created via break in case multiple breaks
    (if brkobjlst
      (progn
        (setq tmppt brkptS) ; use only one of the pair of breakpoints
        ;;  if pt not on object x, switch objects
        (if (not (numberp (vl-catch-all-apply
                            'vlax-curve-getdistatpoint (list obj2break tmppt))))
          (progn ; find the one that pt is on
            (setq idx (length brkobjlst))
            (while (and (not (minusp (setq idx (1- idx))))
                        (setq obj (nth idx brkobjlst))
                        (if (numberp (vl-catch-all-apply
                                       'vlax-curve-getdistatpoint (list obj tmppt)))
                          (null (setq obj2break obj)) ; switch objects, null causes exit
                          t
                        )
                   )
            )
          )
        )
      )
    )
    ;| ;; ver 2.0 fix - removed this code as there are cases where the break point
       ;; is at the end of a line which is not a valid break but does no harm
    (if (and brkobjlst idx (minusp idx)
             (null (alert (strcat "Error - point not on object"
                                  "\nPlease report this error to"
                                  "\n   CAB at TheSwamp.org"))))
      (exit)
    )
    |;
    ;; (if (equal (if (null a)(setq a (car(entsel"\nTest Ent"))) a) ent) (princ)) ; debug CAB  -------------

    ;;  Handle any objects that can not be used with the Break command
    ;;  using one point, gap of 0.01 is used
    (setq closedobj (vl-catch-all-apply 'vlax-curve-isclosed (list obj2break)))
    (if closedobj (if (vl-catch-all-error-p closedobj) (setq closedobj nil)))
    (if GapFlg ; gap > 0
      (if closedobj
        (progn ; need to break a closed object
          (setq brkpt2 (vlax-curve-getPointAtDist obj2break
                     (- (vlax-curve-getDistAtPoint obj2break brkptE) 0.01)))
          (vl-cmdf "._break" obj2break "_non" (trans brkpt2 0 1)
                   "_non" (trans brkptE 0 1))
          (and (= "CIRCLE" enttype) (setq enttype "ARC"))
          (setq BrkptE brkpt2)
        )
      )
      ;;  single breakpoint ----------------------------------------------------
      (setq brkptS brkpts brkptE brkpts)
    ) ; endif
    
    ;; (if (null brkptE) (princ)) ; debug
    
    (setq LastEnt (GetLastEnt))
    (if (not (vlax-erased-p obj2break)) (vl-cmdf "_.break" obj2break "_non" (trans brkptS 0 1) "_non" (trans brkptE 0 1)))
    (if closedobj
      (cond
        ( (equal brkptS (vlax-curve-getstartpoint (if (entnext LastEnt) (entnext LastEnt) obj2break)) 0.2)
          (setq postprocesslst (cons (list brkptS (if (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (+ (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)) (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (+ (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)) (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (- (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)))) postprocesslst))
        )
        ( (equal brkptS (vlax-curve-getendpoint (if (entnext LastEnt) (entnext LastEnt) obj2break)) 0.2)
          (setq postprocesslst (cons (list brkptS (if (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (+ (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)) (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (+ (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)) (vlax-curve-getpointatparam (if (entnext LastEnt) (entnext LastEnt) obj2break) (- (vlax-curve-getparamatpoint (if (entnext LastEnt) (entnext LastEnt) obj2break) (vlax-curve-getclosestpointto (if (entnext LastEnt) (entnext LastEnt) obj2break) brkptE)) 0.1)))) postprocesslst))
        )
      )
    )
    (and *BrkVerbose* (princ (setq *brkcnt* (1+ *brkcnt*))) (princ "\r"))
    (and (= "CIRCLE" enttype) (setq enttype "ARC"))
    (if (not (eq LastEnt (entlast))) ; new object was created
      (if closedobj
        (if (not (vlax-erased-p obj2break))
          (setq brkobjlst (cons (entlast) brkobjlst))
          (progn
            (setq brkobjlst (vl-remove obj2break brkobjlst))
            (entdel (setq obj2break (entlast)))
            (setq LastEnt (entlast))
            (entdel obj2break)
            (setvar 'qaflags 1)
            (vl-cmdf "_.JOIN" obj2break LastEnt "")
            (setvar 'qaflags 0)
            (setq brkobjlst (cons (entlast) brkobjlst))
          )
        )
        (if (not (vlax-erased-p obj2break))
          (setq brkobjlst (cons (entlast) brkobjlst))
          (progn
            (setq brkobjlst (vl-remove obj2break brkobjlst))
            (entdel (setq obj2break (entlast)))
            (setq brkobjlst (cons (entlast) brkobjlst))
            (entdel obj2break)
            (setq brkobjlst (cons obj2break brkobjlst))
          )
        )
      )
      (if closedobj
        (if (not (vlax-erased-p obj2break))
          (if (not (vl-position obj2break brkobjlst)) (setq brkobjlst (cons obj2break brkobjlst)))
        )
      )
    )
  )
  )
  ) ; endif brkptlst
  
) ; defun break_obj

;;====================================
;;  CAB - get last entity in datatbase
;;====================================

(defun GetLastEnt ( / ename result )
  (if (setq result (entlast))
    (while (setq ename (entnext result))
      (setq result ename)
    )
  )
  result
)

;;===================================
;;  CAB - return a list of new enames
;;===================================

(defun GetNewEntities (ename / new)
  (cond
    ((null ename) (alert "Ename nil"))
    ((eq 'ENAME (type ename))
      (while (setq ename (entnext ename))
        (if (not (vlax-erased-p ename)) (setq new (cons ename new)))
      )
    )
    ((alert "Ename wrong type."))
  )
  new
)

;; Point -> Ellipse Parameter  -  Lee Mac
;; Returns the ellipse parameter for the given point
;; dxf  -  Ellipse DXF data (DXF groups 10, 11, 40, 210)
;; pnt  -  WCS Point on Ellipse
;; Uses relationship: ratio*tan(param) = tan(angle)

(defun ELL:point->param ( dxf pnt / ang ocs )
    (setq ocs (cdr (assoc 210 dxf))
          ang (- (angle (trans (cdr (assoc 10 dxf)) 0 ocs) (trans pnt 0 ocs))
                 (angle '(0.0 0.0) (trans (cdr (assoc 11 dxf)) 0 ocs))
              )
    )
    (atan (sin ang) (* (cdr (assoc 40 dxf)) (cos ang)))
)
  
;; Point -> Arc Parameter  -  M.R.
;; Returns the arc parameter for the given point
;; dxf  -  Arc DXF data (DXF groups 10, 210)
;; pnt  -  WCS Point on Arc

(defun ARC:point->param ( dxf pnt / ang ocs )
    (setq ocs (cdr (assoc 210 dxf))
          ang (angle (cdr (assoc 10 dxf)) (trans pnt 0 ocs))
    )
    ang
)

;; Bulge to Arc  -  Lee Mac - mod by M.R.
;; p1 - start vertex
;; p2 - end vertex
;; b  - bulge
;; Returns: (<center> <start angle> <end angle> <radius>)

(defun LM:Bulge->Arc ( p1 p2 b / a c r )
    (setq a (* 2 (atan (abs b)))
          r (abs (/ (distance p1 p2) 2 (sin a)))
          c (if (minusp b) (polar p2 (+ (- (/ pi 2) a) (angle p2 p1)) r) (polar p1 (+ (- (/ pi 2) a) (angle p1 p2)) r))
    )
    (list c (angle c p1) (angle c p2) r)
)

;; Arc to Bulge  -  Lee Mac - mod by M.R.
;; c     - center
;; a1,a2 - start, end angle
;; r     - radius
;; lw    - LWPOLYLINE ename
;; Returns: (<vertex> <bulge> <vertex>)

(defun LM:Arc->Bulge ( c a1 a2 r lw / data1 data2 )

  (if (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a2 r)) 3e-2)) (cdr (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a1 r)) 3e-2)) (entget lw))))
    (setq data1 (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a1 r)) 3e-2)) (reverse (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a2 r)) 3e-2)) (reverse (entget lw))))))
    (setq data2 (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a2 r)) 3e-2)) (reverse (vl-member-if '(lambda ( x ) (equal (cdr x) (mapcar '+ '(0.0 0.0) (polar c a1 r)) 3e-2)) (reverse (entget lw))))))
  )

  (list
    (if (and data1 (not data2))
      (polar c a1 r)
      (polar c a2 r)
    )
    (if (and data1 (not data2))
      (if (minusp (cdr (assoc 42 (reverse data1)))) (if (equal (cdr (assoc 42 (reverse data1))) (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))) 3e-2) (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))) (/ 1 (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))))) (if (equal (cdr (assoc 42 (reverse data1))) (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )) 3e-2) (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )) (/ 1 (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )))))
      (if (minusp (cdr (assoc 42 (reverse data2)))) (if (equal (cdr (assoc 42 (reverse data2))) (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))) 3e-2) (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))) (/ 1 (- (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) ))))) (if (equal (cdr (assoc 42 (reverse data2))) (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )) 3e-2) (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )) (/ 1 (abs ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )))))
    ) ;;; This should be either equal to abs or (- abs) of ( (lambda ( x ) (/ (sin x) (cos x))) (/ (rem (+ pi pi (- a2 a1)) (+ pi pi)) 4.0) )
    (if (and data1 (not data2))
      (polar c a2 r)
      (polar c a1 r)
    )
  )
)

  ;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  ;;         S T A R T  S U B R O U T I N E   H E R E              
  ;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   
    (setq LastEntInDatabase (GetLastEnt))
    (if (and ss2brk ss2brkwith)
    (progn
      (setq oc 0
            ss2brkwithList (ssget->vla-list ss2brkwith))
      (if (> (* (sslength ss2brk) (length ss2brkwithList)) 5000)
        (setq *BrkVerbose* t)
      )
      (and *BrkVerbose*
          (princ (strcat "Objects to be Checked: "
          (itoa (* (sslength ss2brk) (length ss2brkwithList))) "\n")))
      ;;  CREATE a list of entity & it's break points
      (foreach obj (ssget->vla-list ss2brk) ; check each object in ss2brk
        (if (not (onlockedlayer (vlax-vla-object->ename obj)))
          (progn
            (setq lst nil)
            ;; check for break pts with other objects in ss2brkwith
            (foreach intobj ss2brkwithList
              (if (and (or self (not (equal obj intobj)))
                       (setq intpts (get_interpts obj intobj))
                  )
                (setq lst (append (list->3pair intpts) lst)) ; entity w/ break points
              )
              (and *BrkVerbose* (princ (strcat "Objects Checked: " (itoa (setq oc (1+ oc))) "\r")))
            )
            (if lst
              (setq masterlist (cons (cons (vlax-vla-object->ename obj) lst) masterlist))
            )
          )
        )
      )

      
      (and *BrkVerbose* (princ "\nBreaking Objects.\n"))
      (setq *brkcnt* 0) ; break counter
      ;;  masterlist = ((ent brkpts)(ent brkpts)...)
      (if masterlist
        (foreach obj2brk masterlist
          (break_obj (car obj2brk) (cdr obj2brk) Gap)
          (setq *brkcnt* (1+ *brkcnt*))
        )
      )
    )
  )
;;==============================================================
  (and (zerop *brkcnt*) (princ "\nNone to be broken."))
  (setq *BrkVerbose* nil)
  (setq NewEnts (GetNewEntities LastEntInDatabase)) ; return list of enames of new objects
  (setq ss (ssadd))
  (foreach ent NewEnts
    (ssadd ent ss)
  )
  (sssetfirst nil ss)
  (princ)
)

;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;;      E N D   O F    M A I N   S U B R O U T I N E             
;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;;           M A I N   S U B   F U N C T I O N S                 
;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

;;======================
;;  Redraw ss with mode 
;;======================

(defun ssredraw (ss mode / i num)
  (setq i -1)
  (while (setq ename (ssname ss (setq i (1+ i))))
    (redraw (ssname ss i) mode)
  )
)

;;===========================================================================
;;  get all objects touching entities in the sscross                         
;;  limited obj types to "LINE,ARC,SPLINE,LWPOLYLINE,POLYLINE,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"
;;  returns a list of enames
;;===========================================================================

(defun gettouching (sscross / ss lst lstb lstc objl) 
  (and
    (setq lstb (vl-remove-if 'listp (mapcar 'cadr (ssnamex sscross)))
          objl (mapcar 'vlax-ename->vla-object lstb)
    )
    (setq
      ss (ssget "_A" (list (cons 0 "LINE,ARC,SPLINE,*POLYLINE,CIRCLE,ELLIPSE,RAY,XLINE,HELIX")
                           (cons 410 (getvar "ctab"))))
    )
    (setq lst (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss))))
    (setq lst (mapcar 'vlax-ename->vla-object lst))
    (mapcar
      '(lambda (x)
         (mapcar
           '(lambda (y)
              (if (not
                    (vl-catch-all-error-p
                      (vl-catch-all-apply
                        '(lambda ()
                           (vlax-safearray->list
                             (vlax-variant-value
                               (vla-intersectwith y x acextendnone)
                             ))))))
                (setq lstc (cons (vlax-vla-object->ename x) lstc))
              )
            ) objl)
       ) lst)
  )
  lstc
)

(defun oldheavyfit2lw ( pl / ss pea )
  (vl-cmdf "_.EXPLODE" pl)
  (while (< 0 (getvar 'cmdactive)) (vl-cmdf ""))
  (setq ss (ssget "_P"))
  (setq pea (getvar 'peditaccept))
  (setvar 'peditaccept 1)
  (vl-cmdf "_.PEDIT" "_M" ss "" "_J" "_J" "_E" 0.0)
  (while (< 0 (getvar 'cmdactive))
    (vl-cmdf "")
  )
  (setvar 'peditaccept pea)
  (entlast)
)

(defun 3dpoly2lwpoly ( pol / unit mxv v^v transptucs transptwcs entmakelwpoly3dpts vert pt ptlst )

  ; transptucs & transptwcs by M.R. (Marko Ribar, d.i.a.)
  ; arguments : 
  ; pt - point to be transformed from WCS to imaginary UCS with transptucs and from imaginary UCS to WCS with transptwcs
  ; pt1 - origin of imaginary UCS
  ; pt2 - point to define X axis of imaginary UCS (vector pt1-pt2 represents X axis)
  ; pt3 - point to define Y axis of imaginary UCS (vector pt1-pt3 represents Y axis)
  ; important note : angle between X and Y axises of imaginary UCS must always be 90 degree for correct transformation calculation

  ;; Unit Vector - M.R.
  ;; Args: v - vector in R^n

  (defun unit ( v )
    (mapcar '(lambda ( x ) (/ x (distance '(0.0 0.0 0.0) v))) v)
  )

  ;; Matrix x Vector - Vladimir Nesterovsky
  ;; Args: m - nxn matrix, v - vector in R^n

  (defun mxv ( m v )
    (mapcar '(lambda ( r ) (apply '+ (mapcar '* r v))) m)
  )

  ;; Vector Cross Product - Lee Mac
  ;; Args: u,v - vectors in R^3

  (defun v^v ( u v )
    (list
      (- (* (cadr u) (caddr v)) (* (cadr v) (caddr u)))
      (- (* (car  v) (caddr u)) (* (car  u) (caddr v)))
      (- (* (car  u) (cadr  v)) (* (car  v) (cadr  u)))
    )
  )

  (defun transptucs ( pt p1 p2 p3 / ux uy uz )
    (setq uz (unit (v^v (mapcar '- p2 p1) (mapcar '- p3 p1))))
    (setq ux (unit (mapcar '- p2 p1)))
    (setq uy (unit (mapcar '- p3 p1)))
    
    (mxv (list ux uy uz) (mapcar '- pt p1))
  )

  (defun transptwcs ( pt pt1 pt2 pt3 / pt1n pt2n pt3n )
    (setq pt1n (transptucs '(0.0 0.0 0.0) pt1 pt2 pt3))
    (setq pt2n (transptucs '(1.0 0.0 0.0) pt1 pt2 pt3))
    (setq pt3n (transptucs '(0.0 1.0 0.0) pt1 pt2 pt3))
    (transptucs pt pt1n pt2n pt3n)
  )

  (defun entmakelwpoly3dpts ( ptlst 70dxfflag lay dxf62 dxf420 / ux uy uz uptlst )
    (setq uz (unit (v^v (mapcar '- (cadr ptlst) (car ptlst)) (mapcar '- (caddr ptlst) (car ptlst)))))
    (if (equal uz '(0.0 0.0 1.0) 1e-8) (setq ux '(1.0 0.0 0.0) uy '(0.0 1.0 0.0)))
    (if (equal uz '(0.0 0.0 -1.0) 1e-8) (setq ux '(-1.0 0.0 0.0) uy '(0.0 1.0 0.0)))
    (if (not (or (equal uz '(0.0 0.0 1.0) 1e-8) (equal uz '(0.0 0.0 -1.0) 1e-8))) (setq ux (unit (v^v '(0.0 0.0 1.0) uz))))
    (if (not uy) (setq uy (unit (v^v uz ux))))
    (setq uptlst (mapcar '(lambda ( p ) (transptucs p '(0.0 0.0 0.0) ux uy)) ptlst))
    (entmakex
      (append
        (vl-remove nil
          (list
            '(0 . "LWPOLYLINE")
            (cons 8 lay)
            '(100 . "AcDbEntity")
            '(100 . "AcDbPolyline")
            (cons 90 (length uptlst))
            (cons 70 70dxfflag)
            (cons 38 (caddar uptlst))
            (if dxf62 dxf62)
            (if dxf420 dxf420)
            (if dxf430 dxf430)
          )
        )
        (mapcar '(lambda ( x ) (list 10 (car x) (cadr x))) uptlst)
        (list (cons 210 uz))
      )
    )
  )

  (if (and pol (= (cdr (assoc 100 (cdr (member (assoc 100 (entget pol)) (entget pol))))) "AcDb3dPolyline"))
    (progn
      (setq vert (entnext pol))
      (while (= (cdr (assoc 0 (entget vert))) "VERTEX")
        (setq pt (cdr (assoc 10 (entget vert))))
        (setq ptlst (cons pt ptlst))
        (setq vert (entnext vert))
      )
      (setq ptlst (reverse ptlst))
      (entmakelwpoly3dpts ptlst (- (cdr (assoc 70 (entget pol))) 8) (cdr (assoc 8 (entget pol))) (assoc 62 (entget pol)) (assoc 420 (entget pol)))
    )
  )
)

;=========================================
; process-ss main sub. for sel.sets - M.R.
;=========================================

(defun process-ss ( ss )
  (foreach el (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss)))
    (cond
      ( (and
          (eq (cdr (assoc 0 (entget el))) "POLYLINE")
          (or
            (eq (cdr (assoc 70 (entget el))) 0)
            (eq (cdr (assoc 70 (entget el))) 1)
            (eq (cdr (assoc 70 (entget el))) 128)
            (eq (cdr (assoc 70 (entget el))) 129)
          )
        )
        (vl-cmdf "_.CONVERTPOLY" "_L" el "")
      )
      ( (and
          (eq (cdr (assoc 0 (entget el))) "POLYLINE")
          (or
            (eq (cdr (assoc 70 (entget el))) 2)
            (eq (cdr (assoc 70 (entget el))) 3)
            (eq (cdr (assoc 70 (entget el))) 130)
            (eq (cdr (assoc 70 (entget el))) 131)
          )
        )
        (ssdel el ss)
        (setq ell (oldheavyfit2lw el))
        (ssadd ell sn)
        (if (not (vlax-erased-p el))
          (entdel el)
        )
      )
      ( (and
          (vlax-curve-isplanar el)
          (eq (cdr (assoc 0 (entget el))) "POLYLINE")
          (or
            (eq (cdr (assoc 70 (entget el))) 8)
            (eq (cdr (assoc 70 (entget el))) 9)
          )
        )
        (cond
          ( (eq (cdr (assoc 0 (entget (entnext (entnext el))))) "SEQEND")
            (ssdel el ss)
            (if (not (vlax-erased-p el))
              (entdel el)
            )
          )
          ( (eq (cdr (assoc 0 (entget (entnext (setq v2 (entnext (setq v1 (entnext el)))))))) "SEQEND")
            (setq ell (entmakex (vl-remove nil (list '(0 . "LINE") (assoc 8 (entget el)) (if (assoc 62 (entget el)) (assoc 62 (entget el))) (if (assoc 420 (entget el)) (assoc 420 (entget el))) (if (assoc 430 (entget el)) (assoc 430 (entget el))) (assoc 10 (entget v1)) (cons 11 (cdr (assoc 10 (entget v2))))))))
            (ssadd ell sn)
            (ssdel el ss)
            (if (not (vlax-erased-p el))
              (entdel el)
            )
          )
          ( t
            (setq ell (3dpoly2lwpoly el))
            (ssadd ell sn)
            (ssdel el ss)
            (if (not (vlax-erased-p el))
              (entdel el)
            )
          )
        )
      )
      ( (and
          (eq (cdr (assoc 0 (entget el))) "POLYLINE")
          (not
            (or
              (eq (cdr (assoc 70 (entget el))) 0)
              (eq (cdr (assoc 70 (entget el))) 1)
              (eq (cdr (assoc 70 (entget el))) 8)
              (eq (cdr (assoc 70 (entget el))) 9)
              (eq (cdr (assoc 70 (entget el))) 128)
              (eq (cdr (assoc 70 (entget el))) 129)
            )
          )
        )
        (vl-cmdf "_.SPLINEDIT" el "")
        (while (> (getvar 'cmdactive) 0) (vl-cmdf ""))
        (setq ell (entlast))
        (ssdel el ss)
        (ssadd ell sn)
      )
      ( (eq (cdr (assoc 0 (entget el))) "XLINE")
        (setq ell (entmakex (vl-remove nil (list '(0 . "LINE") (assoc 8 (entget el)) (if (assoc 62 (entget el)) (assoc 62 (entget el))) (if (assoc 420 (entget el)) (assoc 420 (entget el))) (if (assoc 430 (entget el)) (assoc 430 (entget el))) (cons 10 (vlax-curve-getpointatparam el -1e+14)) (cons 11 (vlax-curve-getpointatparam el 1e+14))))))
        (ssdel el ss)
        (ssadd ell sn)
        (if (not (vlax-erased-p el))
          (entdel el)
        )
      )
      ( (eq (cdr (assoc 0 (entget el))) "RAY")
        (setq ell (entmakex (vl-remove nil (list '(0 . "LINE") (assoc 8 (entget el)) (if (assoc 62 (entget el)) (assoc 62 (entget el))) (if (assoc 420 (entget el)) (assoc 420 (entget el))) (if (assoc 430 (entget el)) (assoc 430 (entget el))) (cons 10 (vlax-curve-getstartpoint el)) (cons 11 (vlax-curve-getpointatparam el 1e+14))))))
        (ssdel el ss)
        (ssadd ell sn)
        (if (not (vlax-erased-p el))
          (entdel el)
        )
      )
      ( t
        (ssdel el ss)
        (ssadd el sn)
      )
    )
  )
)

;================================================
;  chain selection - Lee Mac
;================================================

(defun css ( en fz / s df fl in l1 l2 s1 s2 sf vl )

  (setq sf
    (append
      (list
       '(-4 . "<OR")
         '(0 . "LINE,ARC")
         '(-4 . "<AND")
             '(0 . "LWPOLYLINE,SPLINE")
             '(-4 . "<NOT")
               '(-4 . "&=")
               '(70 . 1)
             '(-4 . "NOT>")
         '(-4 . "AND>")
         '(-4 . "<AND")
           '(0 . "POLYLINE")
           '(-4 . "<NOT")
               '(-4 . "&")
               '(70 . 89)
           '(-4 . "NOT>")
           '(-4 . "AND>")
         '(-4 . "<AND")
           '(0 . "ELLIPSE")
           '(-4 . "<OR")
             '(-4 . "<>")
             '(41 . 0.0)
             '(-4 . "<>")
              (cons 42 (+ pi pi))
           '(-4 . "OR>")
         '(-4 . "AND>")
       '(-4 . "OR>")
        (if (= 1 (getvar 'cvport))
          (cons 410 (getvar 'ctab))
         '(410 . "Model")
        )
      )
    )
  )
  
  (if (setq s1 (ssget "_X" sf))
    (if (or en (setq s (ssget "_+.:E:S" sf)))
      (progn
        (setq s2 (ssadd))
        (if s
          (setq en (ssname s 0))
        )
        (setq l1 (list (vlax-curve-getstartpoint en) (vlax-curve-getendpoint en)))
        (repeat (setq in (sslength s1))
          (setq en (ssname s1 (setq in (1- in)))
                vl (cons (list (vlax-curve-getstartpoint en) (vlax-curve-getendpoint en) en) vl)
          )
        )
        (while
          (progn
            (foreach v vl
              (if (vl-some '(lambda ( p ) (or (equal (car v) p fz) (equal (cadr v) p fz))) l1)
                (setq s2 (ssadd (caddr v) s2)
                      l1 (vl-list* (car v) (cadr v) l1)
                      fl t
                )
                (setq l2 (cons v l2))
              )
            )
            fl
          )
          (setq vl l2 l2 nil fl nil)
        )
      )
    )
    (princ "\nNo valid objects found.")
  )
  (sssetfirst nil s2)
  (princ)
)

;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;;          E N D   M A I N    F U N C T I O N S                 
;;+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

;;===============================================
;;   Break all objects selected with each other  
;;===============================================

(defun c:BreakAll-nogap (/ cmd ss sn sx sss ell v1 v2 NewEnts AllEnts tmp postprocesslst s ent dxf10 dxf11 ab ba) 

  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (command)
  (if (= 8 (logand 8 (getvar (quote undoctl))))
    (if command-s
      (command-s "_.undo" "_end")
      (vl-cmdf "_.undo" "_end")
    )
  )
  (if command-s
    (command-s "_.undo" "_mark")
    (vl-cmdf "_.undo" "_mark")
  )
  ;; (or Bgap (setq Bgap 0)) ; default
  ;;  get objects to break
  (if (and (setq ss (ssget "_I" (list (cons 0 "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX")))) (= (type ss) (quote pickset)) (> (sslength ss) 0))
    (progn
      (setq sn (ssadd))
      (process-ss ss)
      (break_with sn sn nil 0)
      (sssetfirst)
      ; ss2break ss2breakwith (flag nil = not to break with self)
      (css (GetLastEnt) 1e-6)
      (if (and (setq sx (ssget "_I")) (= (type sx) (quote pickset)) (> (sslength sx) 0))
        (setq NewEnts (vl-remove-if (function listp) (mapcar (function cadr) (ssnamex sx))))
      )
    )
  )
  (if postprocesslst
    (foreach process postprocesslst
      (cond
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "SPLINE"))
          (entupd (cdr (assoc -1 (entmod (if (and (assoc 11 (entget ent)) (setq dxf11 (car (vl-member-if '(lambda ( x ) (equal (cons 11 (trans (osnap (trans (cadr process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent))))) (subst (cons 11 (car process)) dxf11 (entget ent)) (if (setq dxf10 (car (vl-member-if '(lambda ( x ) (equal (cons 10 (trans (osnap (trans (cadr process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent)))) (subst (cons 10 (car process)) dxf10 (entget ent))))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ELLIPSE"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (ELL:point->param (entget ent) (car process))) (assoc 41 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 41 (entget ent)) (entget ent)) )) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (ELL:point->param (entget ent) (car process))) (assoc 42 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 42 (entget ent)) (entget ent)) )))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ARC"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) (subst (cons 50 (ARC:point->param (entget ent) (car process))) (assoc 50 (entget ent)) (entget ent)) (subst (cons 51 (ARC:point->param (entget ent) (car process))) (assoc 51 (entget ent)) (entget ent)))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "LWPOLYLINE"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) 
            (progn
              (if (zerop (cdr (assoc 42 (entget ent))))
                (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (entget ent)) (entget ent))
                (progn
                  (setq ba (LM:Bulge->Arc (cdr (assoc 10 (entget ent))) (cdr (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent))))) (cdr (assoc 42 (entget ent)))))
                  (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                  (subst (cons 10 (caddr ab)) (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent)))) (subst (cons 42 (cadr ab)) (assoc 42 (entget ent)) (subst (cons 10 (car ab)) (assoc 10 (entget ent)) (entget ent))))
                )
              )
            )
            (progn
              (if (zerop (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent)))))))
                (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (reverse (entget ent))) (entget ent))
                (progn
                  (setq ba (LM:Bulge->Arc (cdr (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent)))))) (cdr (assoc 10 (reverse (entget ent)))) (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))))))
                  (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                  (subst (cons 10 (car ab)) (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 42 (cadr ab)) (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 10 (caddr ab)) (assoc 10 (reverse (entget ent))) (entget ent))))
                )
              )
            )
          )
          ))))
        )
      )
      (cond
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "SPLINE"))
          (entupd (cdr (assoc -1 (entmod (if (and (assoc 11 (entget ent)) (setq dxf11 (car (vl-member-if '(lambda ( x ) (equal (cons 11 (trans (osnap (trans (car process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent))))) (subst (cons 11 (car process)) dxf11 (entget ent)) (if (setq dxf10 (car (vl-member-if '(lambda ( x ) (equal (cons 10 (trans (osnap (trans (car process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent)))) (subst (cons 10 (car process)) dxf10 (entget ent))))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ELLIPSE"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (ELL:point->param (entget ent) (car process))) (assoc 41 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 41 (entget ent)) (entget ent)) )) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (ELL:point->param (entget ent) (car process))) (assoc 42 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 42 (entget ent)) (entget ent)) )))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ARC"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) (subst (cons 50 (ARC:point->param (entget ent) (car process))) (assoc 50 (entget ent)) (entget ent)) (subst (cons 51 (ARC:point->param (entget ent) (car process))) (assoc 51 (entget ent)) (entget ent)))))))
        )
        ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "LWPOLYLINE"))
          (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) 
            (progn
              (if (zerop (cdr (assoc 42 (entget ent))))
                (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (entget ent)) (entget ent))
                (progn
                  (setq ba (LM:Bulge->Arc (cdr (assoc 10 (entget ent))) (cdr (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent))))) (cdr (assoc 42 (entget ent)))))
                  (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                  (subst (cons 10 (caddr ab)) (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent)))) (subst (cons 42 (cadr ab)) (assoc 42 (entget ent)) (subst (cons 10 (car ab)) (assoc 10 (entget ent)) (entget ent))))
                )
              )
            )
            (progn
              (if (zerop (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent)))))))
                (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (reverse (entget ent))) (entget ent))
                (progn
                  (setq ba (LM:Bulge->Arc (cdr (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent)))))) (cdr (assoc 10 (reverse (entget ent)))) (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))))))
                  (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                  (subst (cons 10 (car ab)) (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 42 (cadr ab)) (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 10 (caddr ab)) (assoc 10 (reverse (entget ent))) (entget ent))))
                )
              )
            )
          )
          ))))
        )
      )
    )
  )
  (if (= 8 (logand 8 (getvar (quote undoctl))))
    (if command-s
      (command-s "_.undo" "_end")
      (vl-cmdf "_.undo" "_end")
    )
  )
  (setvar "CMDECHO" cmd)
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;===========================================
;;  Break all - gaps are involved 
;;===========================================

(defun c:BreakAll (/ cmd ss sn ell v1 v2 NewEnts AllEnts tmp) 

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\nSelect objects to break with each other & press enter: ")
  (if
    (and
      (setq ss (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss)
        (setq ss (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss)
          )
        )
        t
      )
    )
    (progn
      (break_with ss ss nil Bgap)
      (sssetfirst)
      ; ss2break ss2breakwith (flag nil = not to break with self)
      (css (GetLastEnt) (+ Bgap 1e-6))
      (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;===========================================
;;  Break a single object with other objects 
;;===========================================

(defun c:BreakObject (/ cmd ss1 ss2 sn ell v1 v2 tmp) 

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )

  ;;  get objects to break
  (prompt "\nSelect single object to break: ")
  (if
    (and
      (setq ss1 (ssget "_+.:E:S:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss1)
        (setq ss1 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss1)
          )
        )
        t
      )
      (not (redraw (ssname ss1 0) 3))
      (not (prompt "\n***  Select object(s) to break with & press enter:  ***"))
      (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (not (redraw (ssname ss1 0) 4))
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss1)))
        (css ent 1e-6)
        (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
      )
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;==========================================
;;  Break many objects with a single object 
;;==========================================

(defun c:BreakWobject (/ cmd ss1 ss2 tmp) 

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\nSelect object(s) to break & press enter: ")
  (if 
    (and
      (setq ss1 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss1)
        (setq ss1 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss1)
          )
        )
        t
      )
      (not (ssredraw ss1 3))
      (not (prompt "\n***  Select single object to break with:  ***"))
      (setq ss2 (ssget "_+.:E:S:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (not (ssredraw ss1 4))
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss1)))
        (css ent 1e-6)
        (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
      )
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;==========================================
;;  Break objects with objects on a layer   
;;==========================================

;;  New 08/01/2008
(defun c:BreakWlayer (/ cmd ss1 ss2 sn ell v1 v2 tmp lay) 

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\n***  Select single object for break layer:  ***")
  
  (if
    (and
      (setq ss2 (ssget "_+.:E:S:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (setq lay (assoc 8 (entget (ssname ss2 0))))
      (setq ss2 (ssget "_X" (list '(0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX")
                                   lay (cons 410 (getvar "ctab")))
                )
      )
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (not (prompt "\nSelect object(s) to break & press enter: "))
      (setq ss1 (ssget (list '(0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX")
                              (cons 8 (strcat "~" (cdr lay)))
                       )
                )
      )
      (progn
        (setq sn (ssadd))
        (process-ss ss1)
        (setq ss1 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss1)
          )
        )
        t
      )
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (css (GetLastEnt) (+ Bgap 1e-6))
      (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;======================================================
;;  Break selected objects with other selected objects  
;;======================================================

(defun c:BreakWith (/ cmd ss1 ss2 sn ell v1 v2 tmp)

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\nBreak selected objects with other selected objects.")
  (prompt "\nSelect object(s) to break & press enter: ")
  (if
    (and
      (setq ss1 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss1)
        (setq ss1 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss1)
          )
        )
        t
      )
      (not (ssredraw ss1 3))
      (not (prompt "\n***  Select object(s) to break with & press enter:  ***"))
      (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (not (ssredraw ss1 4))
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (css (GetLastEnt) (+ Bgap 1e-6))
      (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;=============================================
;;  Break objects touching selected objects    
;;=============================================

(defun c:BreakTouching (/ cmd ss1 ss2 sn ell v1 v2 tmp)

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (setq ss1 (ssadd))
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\nBreak objects touching selected objects.")
  (if
    (and
      (not (prompt "\nSelect object(s) to break & press enter: "))
      (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (progn
        (mapcar '(lambda (x) (ssadd x ss1)) (gettouching ss2))
        t
      )
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss1)))
        (css ent 1e-6)
        (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
      )
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;=================================================
;;  Break touching objects with selected objects   
;;=================================================

;;  New 08/01/2008
(defun c:BreakWithTouching (/ cmd ss1 ss2 sn ell v1 v2 tmp) 

  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )

  ;;  get objects to break
  (prompt "\nBreak objects touching selected objects.")
  (prompt "\nSelect object(s) to break with & press enter: ")
  (if
    (and
      (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        (setq ss1 (ssadd))
        (setq tlst (gettouching ss2))
        (setq tlst (vl-remove-if '(lambda (x)(ssmemb x ss2)) tlst)) ;  remove if in picked ss
        (mapcar '(lambda (x) (ssadd x ss1)) tlst) ; convert to a selection set
        t
      )
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss1)))
        (css ent 1e-6)
        (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
      )
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;==========================================================
;;  Break selected objects with any objects that touch it   
;;==========================================================

(defun c:BreakSelected (/ cmd ss1 ss2 sn ell v1 v2 tmp) 
  
  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (setq ss1 (ssadd))
  (or Bgap (setq Bgap 0)) ; default
  (initget 4) ; no negative numbers
  (if (setq tmp (getdist (strcat "\nEnter Break Gap.<"(rtos Bgap)"> ")))
    (setq Bgap tmp)
  )
  ;;  get objects to break
  (prompt "\nBreak selected objects with any objects that touch it.")
  (if
    (and
      (not (prompt "\nSelect object(s) to break with touching & press enter: "))
      (setq ss2 (ssget '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (progn
        (mapcar '(lambda (x) (ssadd x ss1)) (gettouching ss2))
        t
      )
    )
    (progn
      (break_with ss1 ss2 nil Bgap)
      (sssetfirst)
    ; ss2break ss2breakwith (flag nil = not to break with self)
      (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss1)))
        (css ent 1e-6)
        (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
      )
    )
  )
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (if NewEnts
    (progn
      (setq ss (ssadd))
      (foreach ent NewEnts
        (ssadd ent ss)
      )
      (sssetfirst nil ss)
    )
  )
  (princ)
)

;;==========================================================
;;  Break selected object with any objects that touch it and remove every 
;;  other segment of the broken object, start with selected object 
;;==========================================================

(defun c:BreakRemove (/ cmd ss1 ss2 sn ell v1 v2 tmp entlst ename ent en e dist) 
  
  (vl-cmdf "_.undo" "_begin")
  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (setq ss1 (ssadd))
  (setq Bgap 0
        tmp 1)
  ;;  get object to break
  (prompt "\nBreak selected object with any objects that touch it.")
  (if
    (and 
      (not (prompt "\nSelect object to break with touching & press enter: "))
      (setq ss2 (ssget "_+.:S:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
      (progn
        (setq sn (ssadd))
        (process-ss ss2)
        (setq ss2 (ssadd))
        (if sn
          (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
            (ssadd ent ss2)
          )
        )
        t
      )
      (progn
        (setq ss1 (ssadd))
        (mapcar '(lambda (x) (ssadd x ss1)) (setq touch (gettouching ss2)))
        t
      )
    )
    (setq entlst (break_with ss2 ss1 nil Bgap))
    ; ss2break ss1breakwith (flag nil = not to break with self)
  )
  (if (not (member (setq ename (ssname ss2 0)) entlst))
    (setq entlst (append entlst (list ename)))
  )
  (setq pt (cadr (cadddr (car (ssnamex ss2 0))))) ; pick point for ss2
  (foreach ent entlst ; find the new object that is at the picked spot
    (cond
      ((null dist)
       (if (setq p (vlax-curve-getclosestpointto ent pt))
         (setq dist (distance pt p)
               en ent)))
      ((setq p (vlax-curve-getclosestpointto ent pt))
       (if (< (distance pt p) dist)
         (setq dist (distance pt p)
               en ent)))
    ) 
  )
  (while (not (equal en (setq e (car entlst))))
    (setq entlst (append (cdr entlst) (list e))))
  
  (foreach ent entlst
      (if (zerop (setq tmp (- 1 tmp))) (entdel ent)))
  (setvar "CMDECHO" cmd)
  (vl-cmdf "_.undo" "_end")
  (princ)
)

;; ***************************************************
;;     Function to create a dcl support file if it    
;;       does not exist                               
;;     Usage : (create_dcl "file name")               
;;     Returns : T if successful else nil             
;; ***************************************************

(defun create_Breakdcl (fname / acadfn dcl-rev-check)
  ;;=======================================
  ;;      check revision date Routine          
  ;;=======================================
  (defun dcl-rev-check (fn / rvdate ln lp)
    ;;  revision flag must match exactly and must
    ;;  begin with //
    (setq rvflag "//  Revision Control 01/07/2016 @ 03:00" )
    (if (setq fn (findfile fn))
      (progn ; check rev date
        (setq lp 6) ; read 5 lines
        (setq fn (open fn "r")) ; open file for reading
        (while (> (setq lp (1- lp)) 0)
          (setq ln (read-line fn)) ; get a line from file
          (if (vl-string-search rvflag ln)
            (setq lp 0)
          )
        )
        (close fn) ; close the open file handle
        (if (= lp -1)
          nil ; no new dcl needed
          t ; flag to create new file
        )
      )
      t ; flag to create new file
    )
  )
  (if (null (wcmatch (strcase fname) "*`.DCL"))
    (setq fname (strcat fname ".DCL"))
  )
  (if (dcl-rev-check fname)
    ;; create dcl file in same directory as ACAD.PAT  
    (progn
      (if (setq acadfn (findfile "ACAD.PAT"))
        (progn
          (setq fn (strcat (substr acadfn 1 (- (strlen acadfn) 8)) fname))
          (setq fn (open fn "w"))
        )
        (if (setq acadfn (findfile "DEFAULT.PAT"))
          (progn
            (setq fn (strcat (substr acadfn 1 (- (strlen acadfn) 11)) fname))
            (setq fn (open fn "w"))
          )
        )
      )
      (foreach x (list
                   "// WARNING file will be recreated if you change the next line"
                   rvflag
                   "//BreakAll.DCL"
                   "BreakDCL : dialog { label = \"[ Break All or Some by CAB and M.R.  v2.4 ]\";"
                   "  : text { label = \"--=<  Select type of Break Function needed  >=--\"; "
                   "           key = \"tm\"; alignment = centered; fixed_width = true;}"
                   "    spacer_1;"
                   "    : button { key = \"b1\"; mnemonic = \"T\";  alignment = centered;"
                   "               label = \"Break all objects selected with each other without gap\";} "
                   "    : button { key = \"b2\"; mnemonic = \"T\";  alignment = centered;"
                   "               label = \"Break all objects selected with each other\";} "
                   "    : button { key = \"b3\"; mnemonic = \"T\"; alignment = centered;"
                   "               label = \"Break selected objects with other selected objects\";}"
                   "    : button { key = \"b4\"; mnemonic = \"T\";  alignment = centered;"
                   "               label = \" Break selected objects with any objects that touch it\";}"
                   "    spacer_1;"
                   "  : row { spacer_0;"
                   "    : edit_box {key = \"gap\" ; width = 8; mnemonic = \"G\"; label = \"Gap\"; fixed_width = true;}"
                   "    : button { label = \"Help\"; key = \"help\"; mnemonic = \"H\"; fixed_width = true;} "
                   "    cancel_button;"
                   "    spacer_0;"
                   "  }"
                   "}"
                  ) ; endlist
        (princ x fn)
        (write-line "" fn)
      ) ; end foreach
      (close fn)
      (setq acadfn nil)
      (alert (strcat "\nDCL file created, please restart the routine"
               "\n again if an error occures."))
      t ; return True, file created
    )
    t ; return True, file found
  )
) ; end defun

;;==============================
;;     BreakAll Dialog Routine  
;;==============================

(defun c:MyBreak (/ dclfile dcl# RunDCL BreakHelp cmd txt2num mydonedialog ss sn ell v1 v2 ss1 ss2 postprocesslst ent dxf10 dxf11 s ab ba) 
   ;;  return number or nil
  (defun txt2num (txt / num)
    (if txt
    (or (setq num (distof txt 5))
        (setq num (distof txt 2))
        (setq num (distof txt 1))
        (setq num (distof txt 4))
        (setq num (distof txt 3))
    )
    )
    (if (numberp num)
      num
    )
  )
  (defun mydonedialog (flag)
    (setq DCLgap (txt2num (get_tile "gap")))
    (done_dialog flag)
  )
  (defun RunDCL (/ action)
    (or DCLgap (setq DCLgap 0.1)) ; error trap value
    (action_tile "b1" "(mydonedialog 1)")
    (action_tile "b2" "(mydonedialog 2)")
    (action_tile "b3" "(mydonedialog 3)")
    (action_tile "b4" "(mydonedialog 4)")
    (action_tile "gap" "(setq DCLgap (txt2num value$))")
    (set_tile "gap" (rtos DCLgap))
    (action_tile "help" "(BreakHelp)")
    (action_tile "cancel" "(done_dialog 0)")
    (setq action (start_dialog))
    (or DCLgap (setq DCLgap 0.1)) ; error trap value
    (setq DCLgap (max DCLgap 0)) ; nu negative numbers
    
    (cond
      ( (= action 1) ; BreakAll
        (vl-cmdf "_.undo" "_begin")
        ;;  get objects to break
        (prompt "\nSelect objects to break with each other without gaps & press enter: ")
        (if (setq ss (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
          (progn
            (setq sn (ssadd))
            (process-ss ss)
            (if sn
              (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
                (ssadd ent ss)
              )
            )
            (break_with ss ss nil 0)
            (sssetfirst)
            ; ss2break ss2breakwith (flag nil = not to break with self)
            (css (GetLastEnt) (+ DCLgap 1e-6))
            (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
          )
        )
        (if postprocesslst
          (foreach process postprocesslst
            (cond
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "SPLINE"))
                (entupd (cdr (assoc -1 (entmod (if (and (assoc 11 (entget ent)) (setq dxf11 (car (vl-member-if '(lambda ( x ) (equal (cons 11 (trans (osnap (trans (cadr process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent))))) (subst (cons 11 (car process)) dxf11 (entget ent)) (if (setq dxf10 (car (vl-member-if '(lambda ( x ) (equal (cons 10 (trans (osnap (trans (cadr process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent)))) (subst (cons 10 (car process)) dxf10 (entget ent))))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ELLIPSE"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (ELL:point->param (entget ent) (car process))) (assoc 41 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 41 (entget ent)) (entget ent)) )) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (ELL:point->param (entget ent) (car process))) (assoc 42 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 42 (entget ent)) (entget ent)) )))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ARC"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) (subst (cons 50 (ARC:point->param (entget ent) (car process))) (assoc 50 (entget ent)) (entget ent)) (subst (cons 51 (ARC:point->param (entget ent) (car process))) (assoc 51 (entget ent)) (entget ent)))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (cadr process) 0 1) (list -0.001 -0.001)) (mapcar '+ (trans (cadr process) 0 1) (list 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "LWPOLYLINE"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (cadr process) 0 1) "_end") 1 0) 1e-6) 
                  (progn
                   (if (zerop (cdr (assoc 42 (entget ent))))
                     (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (entget ent)) (entget ent))
                      (progn
                        (setq ba (LM:Bulge->Arc (cdr (assoc 10 (entget ent))) (cdr (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent))))) (cdr (assoc 42 (entget ent)))))
                        (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                        (subst (cons 10 (caddr ab)) (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent)))) (subst (cons 42 (cadr ab)) (assoc 42 (entget ent)) (subst (cons 10 (car ab)) (assoc 10 (entget ent)) (entget ent))))
                      )
                    )
                  )
                  (progn
                    (if (zerop (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent)))))))
                      (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (reverse (entget ent))) (entget ent))
                      (progn
                        (setq ba (LM:Bulge->Arc (cdr (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent)))))) (cdr (assoc 10 (reverse (entget ent)))) (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))))))
                        (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                        (subst (cons 10 (car ab)) (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 42 (cadr ab)) (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 10 (caddr ab)) (assoc 10 (reverse (entget ent))) (entget ent))))
                      )
                    )
                  )
                )
                ))))
              )
            )
            ;|
            (if (and s (= (type s) 'pickset) (> (sslength s) 0))
              (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))))
            )
            |;
            (cond
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "SPLINE"))
                (entupd (cdr (assoc -1 (entmod (if (and (assoc 11 (entget ent)) (setq dxf11 (car (vl-member-if '(lambda ( x ) (equal (cons 11 (trans (osnap (trans (car process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent))))) (subst (cons 11 (car process)) dxf11 (entget ent)) (if (setq dxf10 (car (vl-member-if '(lambda ( x ) (equal (cons 10 (trans (osnap (trans (car process) 0 1) "_end") 1 0)) x 1e-6)) (entget ent)))) (subst (cons 10 (car process)) dxf10 (entget ent))))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ELLIPSE"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (ELL:point->param (entget ent) (car process))) (assoc 41 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 41 (entget ent))) 0.05) (subst (cons 41 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 41 (entget ent)) (entget ent)) )) (cond ( (equal (ELL:point->param (entget ent) (car process)) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (ELL:point->param (entget ent) (car process))) (assoc 42 (entget ent)) (entget ent)) ) ( (equal (+ (* 2 pi) (ELL:point->param (entget ent) (car process))) (cdr (assoc 42 (entget ent))) 0.05) (subst (cons 42 (+ (* 2 pi) (ELL:point->param (entget ent) (car process)))) (assoc 42 (entget ent)) (entget ent)) )))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "ARC"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) (subst (cons 50 (ARC:point->param (entget ent) (car process))) (assoc 50 (entget ent)) (entget ent)) (subst (cons 51 (ARC:point->param (entget ent) (car process))) (assoc 51 (entget ent)) (entget ent)))))))
              )
              ( (and (setq s (ssget "_C" (mapcar '+ (trans (car process) 0 1) (list -0.001 -0.001 -0.001)) (mapcar '+ (trans (car process) 0 1) (list 0.001 0.001 0.001)))) (vl-some '(lambda ( x ) (if (ssmemb x ss) (setq ent x))) (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))) (= (cdr (assoc 0 (entget ent))) "LWPOLYLINE"))
                (entupd (cdr (assoc -1 (entmod (if (equal (vlax-curve-getstartpoint ent) (trans (osnap (trans (car process) 0 1) "_end") 1 0) 1e-6) 
                  (progn
                    (if (zerop (cdr (assoc 42 (entget ent))))
                      (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (entget ent)) (entget ent))
                      (progn
                        (setq ba (LM:Bulge->Arc (cdr (assoc 10 (entget ent))) (cdr (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent))))) (cdr (assoc 42 (entget ent)))))
                        (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                        (subst (cons 10 (caddr ab)) (assoc 10 (cdr (member (assoc 10 (entget ent)) (entget ent)))) (subst (cons 42 (cadr ab)) (assoc 42 (entget ent)) (subst (cons 10 (car ab)) (assoc 10 (entget ent)) (entget ent))))
                      )
                    )
                  )
                  (progn
                    (if (zerop (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent)))))))
                      (subst (cons 10 (trans (car process) 0 ent)) (assoc 10 (reverse (entget ent))) (entget ent))
                      (progn
                        (setq ba (LM:Bulge->Arc (cdr (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent)))))) (cdr (assoc 10 (reverse (entget ent)))) (cdr (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))))))
                        (setq ab (LM:Arc->Bulge (car ba) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (cadr ba)) (if (equal (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba) 0.05) (angle (car ba) (mapcar '+ '(0.0 0.0) (trans (car process) 0 ent))) (caddr ba)) (cadddr ba) ent))
                        (subst (cons 10 (car ab)) (assoc 10 (cdr (member (assoc 10 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 42 (cadr ab)) (assoc 42 (cdr (member (assoc 42 (reverse (entget ent))) (reverse (entget ent))))) (subst (cons 10 (caddr ab)) (assoc 10 (reverse (entget ent))) (entget ent))))
                      )
                    )
                  )
                )
                ))))
              )
            )
            ;|
            (if (and s (= (type s) 'pickset) (> (sslength s) 0))
              (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex s)))))
            )
            |;
          )
        )
        (vl-cmdf "_.undo" "_end")
        (if NewEnts
          (progn
            (setq ss (ssadd))
            (foreach ent NewEnts
              (ssadd ent ss)
            )
            (sssetfirst nil ss)
          )
        )
        (princ)
      )

      ( (= action 2) ; BreakAll
        (vl-cmdf "_.undo" "_begin")
        ;;  get objects to break
        (prompt "\nSelect objects to break with each other & press enter: ")
        (if (setq ss (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
          (progn
            (setq sn (ssadd))
            (process-ss ss)
            (if sn
              (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
                (ssadd ent ss)
              )
            )
            (break_with ss ss nil DCLgap)
            (sssetfirst)
            ; ss2break ss2breakwith (flag nil = not to break with self)
            (css (GetLastEnt) (+ DCLgap 1e-6))
            (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
          )
        )
        (vl-cmdf "_.undo" "_end")
        (if NewEnts
          (progn
            (setq ss (ssadd))
            (foreach ent NewEnts
              (ssadd ent ss)
            )
            (sssetfirst nil ss)
          )
        )
        (princ)
      )
      
      ( (= action 3) ; BreakWith
         ;;  get objects to break
        (prompt "\nBreak selected objects with other selected objects.")
        (prompt "\nSelect object(s) to break & press enter: ")
        (if
          (and
            (setq ss1 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
            (progn
              (setq sn (ssadd))
              (process-ss ss1)
              (if sn
                (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
                  (ssadd ent ss1)
                )
              )
              t
            )
            (not (ssredraw ss1 3))
            (not (prompt "\n***  Select object(s) to break with & press enter:  ***"))
            (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
            (progn
              (setq sn (ssadd))
              (process-ss ss2)
              (if sn
                (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
                  (ssadd ent ss2)
                )
              )
              t
            )
            (not (ssredraw ss1 4))
          )
          (progn
            (break_with ss1 ss2 nil DCLgap)
            (sssetfirst)
          ; ss1break ss2breakwith (flag nil = not to break with self)
            (css (GetLastEnt) (+ DCLgap 1e-6))
            (setq NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I")))))
          )
        )
        (vl-cmdf "_.undo" "_end")
        (if NewEnts
          (progn
            (setq ss (ssadd))
            (foreach ent NewEnts
              (ssadd ent ss)
            )
            (sssetfirst nil ss)
          )
        )
        (princ)
      )

      ( (= action 4) ; BreakSelected
        (setq ss1 (ssadd))
        ;;  get objects to break
        (prompt "\nBreak selected objects with any objects that touch it.")
        (if
          (and
            (not (prompt "\nSelect object(s) to break with touching & press enter: "))
            (setq ss2 (ssget "_:L" '((0 . "LINE,SPLINE,*POLYLINE,ARC,CIRCLE,ELLIPSE,RAY,XLINE,HELIX"))))
            (progn
              (setq sn (ssadd))
              (process-ss ss2)
              (if sn
                (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex sn)))
                  (ssadd ent ss2)
                )
              )
              t
            )
            (progn
              (mapcar '(lambda (x) (ssadd x ss1)) (gettouching ss2))
              t
            )
          )
          (progn
            (break_with ss2 ss1 nil DCLgap)
            (sssetfirst)
          ; ss2break ss1breakwith (flag nil = not to break with self)
            (foreach ent (vl-remove-if 'listp (mapcar 'cadr (ssnamex ss2)))
              (css ent 1e-6)
              (setq NewEnts (append NewEnts (vl-remove-if 'listp (mapcar 'cadr (ssnamex (ssget "_I"))))))
            )
          )
        )
        (vl-cmdf "_.undo" "_end")
        (if NewEnts
          (progn
            (setq ss (ssadd))
            (foreach ent NewEnts
              (ssadd ent ss)
            )
            (sssetfirst nil ss)
          )
        )
        (princ)
      )
    )
  )

  (defun BreakHelp ()
    (alert
      (strcat
        "BreakAll.lsp	 (c) 2007-2012 Charles Alan Butler\n"
        "(mod) 2016 Marko Milorad Ribar aka ribarm (M.R.)\n\n"
        "This LISP routine will break objects based on the routine you select.\n"
        "It will not break objects on locked layers and objects must have intersections.\n"
        "Object types are limited to LINE,ARC,SPLINE,LWPOLYLINE,POLYLINE,CIRCLE,ELLIPSE,RAY,XLINE,HELIX\n"
        "BreakAll-nogap - Break all objects selected with each other without gaps\n"
        "BreakAll -       Break all objects selected with each other\n"
        "BreakwObject  -  Break many objects with a single object\n"
        "BreakObject -    Break a single object with many objects \n"
        "BreakWith -      Break selected objects with other selected objects\n"
        "BreakTouching -  Break objects touching selected objects\n"
        "BreakSelected -  Break selected objects with any objects that touch it\n"
        "The Gap distance is the total opening created.\n"
        "You may run each routine by entering the function name at the command line.\n"
        "For updates & comments contact Charles Alan Butler AKA CAB at TheSwamp.org.\n")
    )
  )
  
  ;;================================================================
  ;;                    Start of Routine                            
  ;;================================================================

  (setq cmd (getvar "CMDECHO"))
  (setvar "CMDECHO" 0)
  (setq dclfile "BreakAll.dcl")
  (cond
    ((not (create_Breakdcl dclfile))
     (prompt (strcat "\nCannot create " dclfile "."))
    )
    ((< (setq dcl# (load_dialog dclfile)) 0)
     (prompt (strcat "\nCannot load " dclfile "."))
    )
    ((not (new_dialog "BreakDCL" dcl#))
     (prompt (strcat "\nProblem with " dclfile "."))
    )
    ((RunDCL))      ; No DCL problems: fire it up
  )
  (and cmd (setvar "CMDECHO" cmd))
  (princ)
)

;;/'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\.
;;    E n d   O f   F i l e   I f   y o u   A r e   H e r e       
;;/'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\./'\.
(princ)
