#include "stdafx.h"
#include "MinimumSpanningTreeSolver.h"

#include <boost/config.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/kruskal_min_spanning_tree.hpp>

using namespace boost;
using namespace delaunator;

MinimumSpanningTreeSolver::MinimumSpanningTreeSolver(const std::vector<AcGePoint3d>& v)
    : inputPoints(v)
{
}

bool MinimumSpanningTreeSolver::initTriangles()
{
    bool returnValue = false;
    try
    {
        std::vector<double> coords2d;
        coords2d.reserve(inputPoints.size() * 2);
        for (const auto& item : inputPoints)
        {
            coords2d.emplace_back(item.x);
            coords2d.emplace_back(item.y);
        }
        pDelaunator.reset(new Delaunator(coords2d));
        returnValue = true;
    }
    catch (...)
    {
        returnValue = false;
        acutPrintf(_T("\nDelaunaySolver failed"));
    }
    return returnValue;
}

std::vector<std::pair<AcGePoint3d, AcGePoint3d>> MinimumSpanningTreeSolver::solveMinimumSpanningTree()
{
    using EdgeNode = std::pair< size_t, size_t>;
    using Graph = adjacency_list < vecS, vecS, undirectedS, no_property, property < edge_weight_t, double > >;
    using Edge = graph_traits <Graph>::edge_descriptor;

    std::vector<std::pair<AcGePoint3d, AcGePoint3d>> path;
    if (!initTriangles())
        return path;

    std::vector<EdgeNode> edges;
    edges.reserve(pDelaunator->triangles.size() * 3);

    std::vector<double> weights;
    weights.reserve(pDelaunator->triangles.size() * 3);

    for (size_t i = 0; i < pDelaunator->triangles.size(); i += 3)
    {
        const auto t0 = pDelaunator->triangles[i + 0];
        const auto t1 = pDelaunator->triangles[i + 1];
        const auto t2 = pDelaunator->triangles[i + 2];
        //
        edges.push_back({ t0 ,t1 });
        edges.push_back({ t1 ,t2 });
        edges.push_back({ t2 ,t0 });
        //
        weights.push_back({ inputPoints[t0].distanceTo(inputPoints[t1]) });
        weights.push_back({ inputPoints[t1].distanceTo(inputPoints[t2]) });
        weights.push_back({ inputPoints[t2].distanceTo(inputPoints[t0]) });
    }
    std::vector < Edge > spanning_tree;
    Graph g(edges.begin(), edges.end(), weights.begin(), inputPoints.size());
    kruskal_minimum_spanning_tree(g, std::back_inserter(spanning_tree));

    for (const auto& item : spanning_tree)
    {
        path.push_back({ inputPoints[source(item, g)],  inputPoints[target(item, g)] });
    }
    pDelaunator.reset(nullptr);
    inputPoints.clear();
    return path;
}

int MinimumSpanningTreeSolver::minimumSpanningTreeSolverLispFunc()
{
    std::vector<AcGePoint3d> points;
    {
        AcResBufPtr pArgs(acedGetArgs());
        for (resbuf* pTail = pArgs.get(); pTail != nullptr; pTail = pTail->rbnext)
        {
            switch (pTail->restype)
            {
                case RTPOINT:
                case RT3DPOINT:
                {
                    points.push_back(asPnt3d(pTail->resval.rpoint));
                    break;
                }
                default:
                    break;
            }
        }
    }
    if (points.size() > 2)
    {
        MinimumSpanningTreeSolver solver(std::move(points));
        const auto& path = solver.solveMinimumSpanningTree();
        if (path.size())
        {
            AcResBufPtr pRes(acutNewRb(RTLB));
            resbuf* pResTail = pRes.get();
            for (auto& item : path)
            {
                pResTail = pResTail->rbnext = acutNewRb(RTLB);

                pResTail = pResTail->rbnext = acutNewRb(RT3DPOINT);
                memcpy(pResTail->resval.rpoint, asDblArray(item.first), sizeof(pResTail->resval.rpoint));

                pResTail = pResTail->rbnext = acutNewRb(RT3DPOINT);
                memcpy(pResTail->resval.rpoint, asDblArray(item.second), sizeof(pResTail->resval.rpoint));

                pResTail = pResTail->rbnext = acutNewRb(RTLE);
            }
            pResTail = pResTail->rbnext = acutNewRb(RTLE);
            acedRetList(pRes.get());
        }
    }
    return (RSRSLT);
}
