#include "stdafx.h"
#include "TSPSolver.h"
#include <ppl.h>

#include "ortools/constraint_solver/routing.h"
#include "ortools/constraint_solver/routing_enums.pb.h"
#include "ortools/constraint_solver/routing_index_manager.h"
#include "ortools/constraint_solver/routing_parameters.h"


using namespace concurrency;
using namespace operations_research;

int TSPSolver::luprc()
{
    resbuf buf;
    if (acedGetVar(_T("LUPREC"), &buf) == RTNORM)
    {
        if (buf.restype == RTSHORT || buf.restype == RTLONG)
            return buf.resval.rint;
    }
    return 4;
}

double TSPSolver::defaultPrecision()
{
    AcString sd = _T("1");
    int luprec = luprc();
    for (int i = 0; i < luprec; i++)
        sd += _T("0");
    return _wtoi(sd);
}

Point3ds TSPSolver::solveFirstSolutionStrategy(Point3ds inputPoints, double scale, int stratagy)
{
    std::sort(std::execution::par, inputPoints.begin(), inputPoints.end(), [](const AcGePoint3d& a, const AcGePoint3d& b) -> bool
    {
        if (a.y == b.y)
            return a.x < b.x;
        return a.y < b.y;
    });
    std::sort(std::execution::par, inputPoints.begin(), inputPoints.end(), [](const AcGePoint3d& a, const AcGePoint3d& b) -> bool
    {
        if (a.x == b.x)
            return a.y < b.y;
        return a.x < b.x;
    });
    DistMatrix distances = DistMatrix(inputPoints.size(), std::vector<int64_t>(inputPoints.size(), int64_t{ 0 }));
    parallel_for(size_t(0), inputPoints.size(), [&](size_t fromNode)
    {
        for (size_t toNode = 0; toNode < inputPoints.size(); toNode++)
        {
            if (fromNode != toNode)
            {
                distances[fromNode][toNode] = static_cast<int64_t>(inputPoints[fromNode].distanceTo(inputPoints[toNode]) * scale);
            }
        }
    });
    const operations_research::RoutingIndexManager::NodeIndex depot{ 0 };
    RoutingIndexManager manager(inputPoints.size(), 1, depot);
    RoutingModel routing(manager);

    int64_t transit_callback_index = routing.RegisterTransitCallback([&](int64_t from_index, int64_t to_index) -> int64_t
    {
        auto from_node = manager.IndexToNode(from_index).value();
        auto to_node = manager.IndexToNode(to_index).value();
        return distances[from_node][to_node];
    });
    routing.SetArcCostEvaluatorOfAllVehicles(transit_callback_index);
    RoutingSearchParameters searchParameters = DefaultRoutingSearchParameters();

    switch (stratagy)
    {
        case 0:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::UNSET);
            break;
        case 1:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::AUTOMATIC);
            break;
        case 2:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::PATH_CHEAPEST_ARC);
            break;
        case 3:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::PATH_MOST_CONSTRAINED_ARC);
            break;
        case 4:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::EVALUATOR_STRATEGY);
            break;
        case 5:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::SAVINGS);
            break;
        case 6:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::SWEEP);
            break;
        case 7:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::CHRISTOFIDES);
            break;
        case 8:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::ALL_UNPERFORMED);
            break;
        case 9:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::BEST_INSERTION);
            break;
        case 10:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::PARALLEL_CHEAPEST_INSERTION);
            break;
        case 11:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::SEQUENTIAL_CHEAPEST_INSERTION);
            break;
        case 12:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::LOCAL_CHEAPEST_INSERTION);
            break;
        case 13:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::LOCAL_CHEAPEST_COST_INSERTION);
            break;
        case 14:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::GLOBAL_CHEAPEST_ARC);
            break;
        case 15:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::LOCAL_CHEAPEST_ARC);
            break;
        case 16:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::FIRST_UNBOUND_MIN_VALUE);
            break;
        default:
            searchParameters.set_first_solution_strategy(FirstSolutionStrategy::UNSET);
            break;
    }
    Point3ds path;
    const Assignment* solution = routing.SolveWithParameters(searchParameters);
    if (solution != nullptr)
    {
        path.reserve(inputPoints.size());
        int64_t index = routing.Start(0);
        while (routing.IsEnd(index) == false)
        {
            path.emplace_back(inputPoints[manager.IndexToNode(index).value()]);
            int64_t previous_index = index;
            index = solution->Value(routing.NextVar(index));
        }
    }
    return path;
}

Point3ds TSPSolver::solveLocalSearchMetaheuristicStrategy(Point3ds inputPoints, double scale, int stratagy, int64_t timeInSeconds)
{
    std::sort(std::execution::par, inputPoints.begin(), inputPoints.end(), [](const AcGePoint3d& a, const AcGePoint3d& b) -> bool
    {
        if (a.y == b.y)
            return a.x < b.x;
        return a.y < b.y;
    });
    std::sort(std::execution::par, inputPoints.begin(), inputPoints.end(), [](const AcGePoint3d& a, const AcGePoint3d& b) -> bool
    {
        if (a.x == b.x)
            return a.y < b.y;
        return a.x < b.x;
    });
    DistMatrix distances = DistMatrix(inputPoints.size(), std::vector<int64_t>(inputPoints.size(), int64_t{ 0 }));
    parallel_for(size_t(0), inputPoints.size(), [&](size_t fromNode)
    {
        for (size_t toNode = 0; toNode < inputPoints.size(); toNode++)
        {
            if (fromNode != toNode)
            {
                distances[fromNode][toNode] = static_cast<int64_t>(inputPoints[fromNode].distanceTo(inputPoints[toNode]) * scale);
            }
        }
    });
    const operations_research::RoutingIndexManager::NodeIndex depot{ 0 };
    RoutingIndexManager manager(inputPoints.size(), 1, depot);
    RoutingModel routing(manager);

    int64_t transit_callback_index = routing.RegisterTransitCallback([&](int64_t from_index, int64_t to_index) -> int64_t
    {
        auto from_node = manager.IndexToNode(from_index).value();
        auto to_node = manager.IndexToNode(to_index).value();
        return distances[from_node][to_node];
    });
    routing.SetArcCostEvaluatorOfAllVehicles(transit_callback_index);
    RoutingSearchParameters searchParameters = DefaultRoutingSearchParameters();
    searchParameters.mutable_time_limit()->set_seconds(timeInSeconds);

    switch (stratagy)
    {
        case 0:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::UNSET);
            break;
        case 1:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::AUTOMATIC);
            break;
        case 2:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::GREEDY_DESCENT);
            break;
        case 3:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::GUIDED_LOCAL_SEARCH);
            break;
        case 4:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::SIMULATED_ANNEALING);
            break;
        case 5:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::TABU_SEARCH);
            break;
        case 6:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::GENERIC_TABU_SEARCH);
            break;
        default:
            searchParameters.set_local_search_metaheuristic(LocalSearchMetaheuristic::UNSET);
            break;
    }
    Point3ds path;
    const Assignment* solution = routing.SolveWithParameters(searchParameters);
    if (solution != nullptr)
    {
        path.reserve(inputPoints.size());
        int64_t index = routing.Start(0);
        while (routing.IsEnd(index) == false)
        {
            path.emplace_back(inputPoints[manager.IndexToNode(index).value()]);
            int64_t previous_index = index;
            index = solution->Value(routing.NextVar(index));
        }
    }
    return path;
}

int TSPSolver::solveFirstSolutionStrategyLispFunc()
{
    size_t narg = 0;
    int stratagy = 0;
    double scale = 1000.0;
    std::vector<AcGePoint3d> points;
    {//scope to free mem
        AcResBufPtr pArgs(acedGetArgs());
        for (resbuf* pTail = pArgs.get(); pTail != nullptr; pTail = pTail->rbnext)
        {
            switch (pTail->restype)
            {
                case RTSHORT:
                {
                    narg == 0 ? stratagy = pTail->resval.rint : scale = pTail->resval.rint;
                    break;
                }
                case RTLONG:
                {
                    narg == 0 ? stratagy = pTail->resval.rlong : scale = pTail->resval.rlong;
                    break;
                }
                case RTREAL:
                {
                    if (narg == 1)
                        scale = pTail->resval.rreal;
                    break;
                }
                case RTPOINT:
                case RT3DPOINT:
                {
                    points.emplace_back(asPnt3d(pTail->resval.rpoint));
                    break;
                }
                default:
                    break;
            }
            narg++;
        }
    }
    if (points.size())
    {
        if (scale < 0)
            scale = defaultPrecision();
        const auto& path = TSPSolver::solveFirstSolutionStrategy(std::move(points), scale, stratagy);
        if (path.size())
        {
            AcResBufPtr pRes(acutNewRb(RTLB));
            resbuf* pResTail = pRes.get();
            for (auto& item : path)
            {
                pResTail = pResTail->rbnext = acutNewRb(RT3DPOINT);
                memcpy(pResTail->resval.rpoint, asDblArray(item), sizeof(pResTail->resval.rpoint));
            }
            pResTail = pResTail->rbnext = acutNewRb(RTLE);
            acedRetList(pRes.get());
        }
    }
    return (RSRSLT);
}

int TSPSolver::solveLocalSearchMetaheuristicStrategyLispFunc()
{
    size_t narg = 0;
    int stratagy = 0;
    double scale = 1000.0;
    int64_t timeInSeconds = 30;
    std::vector<AcGePoint3d> points;
    {
        AcResBufPtr pArgs(acedGetArgs());
        for (resbuf* pTail = pArgs.get(); pTail != nullptr; pTail = pTail->rbnext)
        {
            switch (pTail->restype)
            {
                case RTSHORT:
                {
                    if (narg == 0)
                        stratagy = pTail->resval.rint;
                    else if (narg == 1)
                        timeInSeconds = pTail->resval.rint;
                    else if (narg == 2)
                        scale = pTail->resval.rint;
                    break;
                }
                case RTLONG:
                {
                    if (narg == 0)
                        stratagy = pTail->resval.rlong;
                    else if (narg == 1)
                        timeInSeconds = pTail->resval.rlong;
                    else if (narg == 2)
                        scale = pTail->resval.rlong;
                    break;
                }
                case RTREAL:
                {
                    if (narg == 0)
                        stratagy = static_cast<int>(pTail->resval.rreal);
                    else if (narg == 1)
                        timeInSeconds = static_cast<int64_t>(pTail->resval.rreal);
                    else if (narg == 2)
                        scale = pTail->resval.rreal;
                    break;
                }
                case RTPOINT:
                case RT3DPOINT:
                {
                    points.emplace_back(asPnt3d(pTail->resval.rpoint));
                    break;
                }
                default:
                    break;
            }
            narg++;
        }
    }
    if (points.size())
    {
        if (scale < 0)
            scale = defaultPrecision();
        if (timeInSeconds < 0)
            timeInSeconds = 10;
        const auto& path = TSPSolver::solveLocalSearchMetaheuristicStrategy(std::move(points), scale, stratagy, timeInSeconds);
        if (path.size())
        {
            AcResBufPtr pRes(acutNewRb(RTLB));
            resbuf* pResTail = pRes.get();
            for (auto& item : path)
            {
                pResTail = pResTail->rbnext = acutNewRb(RT3DPOINT);
                memcpy(pResTail->resval.rpoint, asDblArray(item), sizeof(pResTail->resval.rpoint));
            }
            pResTail = pResTail->rbnext = acutNewRb(RTLE);
            acedRetList(pRes.get());
        }
    }
    return (RSRSLT);
}
