;;;                                                                      ;;;
;;;  file:AlignBMultiple.lsp                                             ;;;
;;;                                                                      ;;;
;;;  The lisp aligns between two curves the block elements and creates   ;;;
;;;  a block containing the lines of the new geometry.                   ;;;
;;;  The origin shape of the block shall be a rectangle, an isosceles    ;;;
;;;  triangle or an isosceles trapezoid.                                 ;;;
;;;  The procedure requires the selection of two lines containing the    ;;;
;;;  origin shape, the two destination curves and the block to be        ;;;
;;;  adapted.                                                            ;;;
;;;  The curves can be line, 2dpolyline, lwpolyline, circle, spline,     ;;;
;;;  arc and ellipse.                                                    ;;;
;;;  In case of large blocks is recommended to divide it into portions.  ;;;
;;;                                                                      ;;;
;;;                                                                      ;;;
;;;                                                                      ;;;
;;;  author: Gian Paolo Cattaneo                                         ;;;
;;;                                                                      ;;;
;;;  version: 1.0  -  09/10/2012                                         ;;;
;;;                                                                      ;;;

(defun c:ALHM (/ LL cmd L1 L2 L3 L4 L1a L1b L1p L2a L2b L2p L3a L3b L3p
                L3rev l4a L4b L4p L4rev block RET b_lay b_col ang lineblock
                d1 d2 d3 d4 Pm1 Pm2 LIN p1 p2 p3 p4 p5 p6 p7 p8 d_off e_t
                Lin_B Lin_BL n_seg k min_L cnt RETNew en_L en_LL L_D ang_L _p0 _p1 _pend a b c d
                esel no_segm verifica_posizione_selezione unit vxv LM:Perpendicular-p _vlax-curve-getpointatdist verifica_forma_origine verifica_forma_destinazione
                reverse_curve LineBlock punti_comuni draw_m1 draw_m2 draw_m3 ang_der p_vic crea_blocco alb_dcl ALB_VECTORS
                ALB_clean ALB_m1 ALB_m2 ALB_m3 LM:Popup crea_dcl_alb
              )

  (defun esel (/ :LL)
    (while (not L1)
      (setvar "errno" 0)
      (setq L1 (car (entsel "\nSelect First Source Line ")))
      (if (= 7 (getvar 'errno))
        (LM:Popup "AlignHM" "No objects selected" (+ 0 48 4096))
      )
      (if (= 'ename (type L1))
        (if (null (wcmatch (cdr (assoc 0 (entget L1))) "LINE"))
          (progn
            (LM:Popup "AlignHM" "Invalid object" (+ 0 48 4096))
            (setq L1 nil)
          )
          (progn
            (setq L1p (vlax-curve-getClosestPointTo L1 (trans (cadr (grread T)) 1 0)))
            (redraw L1 3)
          )
        )
      )
    )
    (while (not L3)
      (setvar "errno" 0)
      (setq L3 (car (entsel "\nSelect First Destination Curve ")))
      (if (= 7 (getvar 'errno))
        (LM:Popup "AlignHM" "No objects selected" (+ 0 48 4096))
      )
      (if (= 'ename (type L3))
        (if (null (wcmatch (cdr (assoc 0 (entget L3))) "LINE,POLYLINE,LWPOLYLINE,CIRCLE,ARC,SPLINE,ELLIPSE"))
          (progn
            (LM:Popup "AlignHM" "Invalid object" (+ 0 48 4096))
            (setq L3 nil)
          )
          (progn
            (setq L3p (vlax-curve-getClosestPointTo L3 (trans (cadr (grread T)) 1 0)))
            (redraw L3 3)
            (grdraw (trans L1p 0 1) (trans L3p 0 1) 2 1)
          )
        )
      )
    )
    (while (not L2)
      (setvar "errno" 0)
      (setq L2 (car (entsel "\nSelect Second Source Line ")))
      (if (= 7 (getvar 'errno))
        (LM:Popup "AlignHM" "No objects selected" (+ 0 48 4096))
      )
      (if (= 'ename (type L2))
        (if (null (wcmatch (cdr (assoc 0 (entget L2))) "LINE"))
          (progn
            (LM:Popup "AlignHM" "Invalid object" (+ 0 48 4096))
            (setq L2 nil)
          )
          (progn
            (setq L2p (vlax-curve-getClosestPointTo L2 (trans (cadr (grread T)) 1 0)))
            (redraw L2 3)
          )
        )
      )
    )
    (setq L1a (cdr (assoc 10 (entget L1))))
    (setq L1b (cdr (assoc 11 (entget L1))))
    (setq L2a (cdr (assoc 10 (entget L2))))
    (setq L2b (cdr (assoc 11 (entget L2))))
    (if (> (distance L1p L1a) (distance L1p L1b)) (setq :LL L1a L1a L1b L1b :LL))
    (if (> (distance L2p L2a) (distance L2p L2b)) (setq :LL L2a L2a L2b L2b :LL))
    (setq Pm1 (polar L1a (angle L1a L2a) (/ (distance L1a L2a) 2)))
    (setq Pm2 (polar L1b (angle L1b L2b) (/ (distance L1b L2b) 2)))
    (verifica_posizione_selezione)
    (verifica_forma_origine)
    (while (not L4)
      (setvar "errno" 0)
      (setq L4 (car (entsel "\nSelect Second Destination Curve ")))
      (if (= 7 (getvar 'errno))
        (LM:Popup "AlignHM" "No objects selected" (+ 0 48 4096))
      )
      (if (= 'ename (type L4))
        (if (null (wcmatch (cdr (assoc 0 (entget L4))) "LINE,POLYLINE,LWPOLYLINE,CIRCLE,ARC,SPLINE,ELLIPSE"))
          (progn
            (LM:Popup "AlignHM" "Invalid object" (+ 0 48 4096))
            (setq L4 nil)
          )
          (progn
            (setq L4p (vlax-curve-getClosestPointTo L4 (trans (cadr (grread T)) 1 0)))
            (redraw L4 3)
            (grdraw (trans L1p 0 1) (trans L3p 0 1) 2 1)
            (grdraw (trans L2p 0 1) (trans L4p 0 1) 2 1)
          )
        )
      )
    )
    (setq d1 (distance L1a L1b))
    (setq d2 (distance L2a L2b))
    (setq d3 (vlax-curve-getDistAtParam L3 (vlax-curve-getEndParam L3)))
    (setq d4 (vlax-curve-getDistAtParam L4 (vlax-curve-getEndParam L4)))
    (reverse_curve)
    (if (= align_m 3) (verifica_forma_destinazione))
    (while (not block)
      (setvar "errno" 0)
      (setq block (car (entsel "\nSelect Hatch To Align ")))
      (if (= 7 (getvar 'errno))
        (LM:Popup "AlignHM" "No objects selected" (+ 0 48 4096))
      )
      (if (= 'ename (type block))
        (if (or (null (wcmatch (cdr (assoc 0 (entget block))) "HATCH")) (= (substr (cdr (assoc 2 (entget block))) 1 5) "SOLID"))
          (progn
            (LM:Popup "AlignHM" "Invalid object" (+ 0 48 4096))
            (setq block nil)
          )
        )
      )
    )
    (redraw L1 4) (redraw L2 4) (redraw L3 4) (redraw L4 4)
    t
  )

  (defun no_segm ()
    (initget 7)
    (setq n_seg (getint "\nEnter number of segments for distribution along destination curves : "))
  )

  (defun verifica_posizione_selezione ()
    (if (or
          (equal Pm1 Pm2 1e-4)
          (and
            (vlax-curve-getDistAtPoint L1 Pm1)
            (vlax-curve-getDistAtPoint L2 Pm2)
          )
        )
      (progn
        (LM:Popup "AlignHM"
          ;;(strcat "I punti di selezione delle linee \"origine\""
          ;;       "\nnon devono essere agli opposti")
          "The selection points of source lines should not be on the opposite"
          (+ 0 48 4096)
        )
        (exit)
      )
    )
  )

  (defun unit ( v / d )
    (if (not (equal (setq d (distance '(0.0 0.0 0.0) v)) 0.0 1e-6))
      (mapcar '(lambda ( x ) (/ x d)) v)
    )
  )

  ;; Vector Dot Product  -  Lee Mac
  ;; Args: u,v - vectors in R^n

  (defun vxv ( u v )
      (apply '+ (mapcar '* u v))
  )

  ;; Perpendicular-p  -  Lee Mac
  ;; Returns T if vectors v1,v2 are perpendicular

  (defun LM:Perpendicular-p ( v1 v2 )
      (equal 0.0 (vxv v1 v2) 0.1)
  )

  (defun _vlax-curve-getpointatdist ( curve dist )
    (cond
      ( (= dist 0.0)
        (vlax-curve-getpointatdist curve dist)
      )
      ( (and (/= dist 0.0) (= (vlax-curve-getparamatdist curve dist) 0.0))
        nil
      )
      ( t
        (vlax-curve-getpointatdist curve dist)
      )
    )
  )

  (defun verifica_forma_origine ( / ip )
    (if
      (or
        (and
          (not (equal L1a L2a 1e-5))
          (not (LM:Perpendicular-p (unit (mapcar '- L1a Pm1)) (unit (mapcar '- Pm2 Pm1))))
        )
        (and
          (not (equal L1b L2b 1e-5))
          (not (LM:Perpendicular-p (unit (mapcar '- L2b Pm2)) (unit (mapcar '- Pm1 Pm2))))
        )
        (if (setq ip (inters l1a l1b l2a l2b nil)) nil nil)
        (if ip
          (and (not (equal l2a ip 1e-5)) (not (equal l2b ip 1e-5)) (not (equal (distance l2b ip) (distance l1b ip) 1e-5)))
          nil
        )
        (if ip
          (and (not (equal l1a ip 1e-5)) (not (equal l1b ip 1e-5)) (not (equal (distance l2a ip) (distance l1a ip) 1e-5)))
          nil
        )
      )
      (progn
        (LM:Popup "AlignHM"
          ;"La forma origine deve essere un rettangolo, un triangolo isoscele o trapezio isoscele"
          "The origin shape should be a rectangle, an isosceles triangle or isosceles trapezoid."
          (+ 0 48 4096)
        )
        (exit)
      )
    )
  )

  (defun verifica_forma_destinazione (/ p_L3 step segm p_L4 n+ toll)
    (setq step (- (/ d3 10.0) 1e-5))
    (setq segm 0.0)
    (repeat 11
      (setq p (trans (_vlax-curve-getpointatdist L3 segm) 0 1))
      (setq p_L3 (cons (list (car p) (cadr p)) p_L3))
      (setq segm (+ segm step))
    )
    (setq p_L3 (reverse p_L3))
    (mapcar '(lambda (a / p)
               (setq p (trans (vlax-curve-getClosestPointTo L4 a) 0 1))
               (setq p_L4 (cons (list (car p) (cadr p)) p_L4))
             )
            p_L3
    )
    (setq p_L4 (reverse p_L4))
    (setq d_off (/ (distance (car p_L3) (car p_L4)) 2))
    (setq toll (/ d_off 100.0))
    (setq n+ 0)
    (mapcar '(lambda (a b)
                     (if (or (equal (/ (distance a b) 2) d_off toll) (< (/ (distance a b) 2) d_off)) (setq n+ (1+ n+)))
             )
            p_L3 p_L4
    )
    (if (/= n+ 11)
      (progn
        (LM:Popup "AlignHM"
          ;"Metodo 3 - Gli oggetti \"destinazione\" devono essere paralleli (offset)"
          "Method 3 - The destination curves must be parallel (offset)"
          (+ 0 48 4096)
        )
        (redraw L1 4) (redraw L2 4) (redraw L3 4) (redraw L4 4)
        (exit)
      )
    )
  )

  (defun reverse_curve (/)
    (setq L3a (vlax-curve-getEndPoint L3))
    (setq L3b (vlax-curve-getStartPoint L3))
    (setq L4a (vlax-curve-getEndPoint L4))
    (setq L4b (vlax-curve-getStartPoint L4))
    (if (> (vlax-curve-getDistAtPoint L3 L3p) (/ d3 2))
      (setq L3rev 1)
      (setq L3rev 0)
    )
    (if (> (vlax-curve-getDistAtPoint L4 L4p) (/ d4 2))
      (setq L4rev 1)
      (setq L4rev 0)
    )
  )

  (defun LineBl (/ n)
    (repeat (setq n (sslength RET))
      (setq LIN (ssname RET (setq n (1- n))))
      (setq Lin_B (cons (list
                          (cdr (assoc 10 (entget LIN)))
                          (cdr (assoc 11 (entget LIN)))
                        )
                        Lin_B
                  )
      )
      (entdel LIN)
    )
  )

  (defun punti_comuni (Lin_B)
    (if (null Lin_BL) (setq Lin_BL Lin_B))
    (setq p1 (car (car Lin_BL)))
    (setq p2 (cadr (car Lin_BL)))
    (setq Lin_BL (cdr Lin_BL))
    (setq p3 (polar p1 ang 1.0))
    (setq p4 (polar p2 ang 1.0))
    (setq p5 (inters L1a L1b p1 p3 nil))
    (setq p6 (inters L2a L2b p1 p3 nil))
    (setq p7 (inters L1a L1b p2 p4 nil))
    (setq p8 (inters L2a L2b p2 p4 nil))
  )

  (defun draw_m1 (/ i n m1 m2 LIN p1 p2 p3 p4 p5 p6 p7 p8 E_L
                    p5a p7a p6a p8a p10 p11
                 )

    (setq m1 (* (/ 1.0 n_seg) (/ d3 d1)))
    (setq m2 (* (/ 1.0 n_seg) (/ d4 d2)))
    (setq n (length Lin_B))
    (if (= e_t 1) (acet-ui-progress-init "" n))
    (setq i 0)
    (repeat n
      (punti_comuni Lin_B)
      (if (= L3rev 0)
        (progn
          (setq p5a (_vlax-curve-getpointatdist L3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p5)))))
          (setq p7a (_vlax-curve-getpointatdist L3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p7)))))
          (if (null p5a) (setq p5a L3a))
          (if (null p7a) (setq p7a L3a))
        )
        (progn
          (setq p5a (_vlax-curve-getpointatdist L3 (- d3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p5))))))
          (setq p7a (_vlax-curve-getpointatdist L3 (- d3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p7))))))
          (if (null p5a) (setq p5a L3b))
          (if (null p7a) (setq p7a L3b))
        )
      )
      (if (= L4rev 0)
        (progn
          (setq p6a (_vlax-curve-getpointatdist L4 (+ (* k (/ d4 n_seg)) (* m2 (distance L2a p6)))))
          (setq p8a (_vlax-curve-getpointatdist L4 (+ (* k (/ d4 n_seg)) (* m2 (distance L2a p8)))))
          (if (null p6a) (setq p6a L4a))
          (if (null p8a) (setq p8a L4a))
        )
        (progn
          (setq p6a (_vlax-curve-getpointatdist L4 (- d4 (+ (* k (/ d4 n_seg)) (* m2 (distance L2a p6))))))
          (setq p8a (_vlax-curve-getpointatdist L4 (- d4 (+ (* k (/ d4 n_seg)) (* m2 (distance L2a p8))))))
          (if (null p6a) (setq p6a L4b))
          (if (null p8a) (setq p8a L4b))
        )
      )
      (setq d (/ (* (distance p1 p5) (distance p5a p6a)) (distance p5 p6)))
      (setq p10 (polar p5a (angle p5a p6a) d))
      (setq d (/ (* (distance p2 p7) (distance p7a p8a)) (distance p7 p8)))
      (setq p11 (polar p7a (angle p7a p8a) d))
      (setq LineBlock (cons (list p10 p11) LineBlock))
      (setq i (1+ i))
      (if (= e_t 1) (acet-ui-progress-safe i))
    )
    (if (= e_t 1) (acet-ui-progress-done))
  )

  (defun draw_m2 (/ m1 n p5a p6a p7a p8a p5b p7b L5 L6 L4# L5# L6# ang1 ang2 i p10 p11)
    (setq m1 (* (/ 1.0 n_seg) (/ d3 d1)))
    (setq n (length Lin_B))
    (if (= e_t 1) (acet-ui-progress-init "" n))
    (setq i 0)
    (repeat n
      (punti_comuni Lin_B)
      (if (= L3rev 0)
        (progn
          (setq p5a (_vlax-curve-getpointatdist L3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p5)))))
          (setq p7a (_vlax-curve-getpointatdist L3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p7)))))
          (if (null p5a) (setq p5a L3a))
          (if (null p7a) (setq p7a L3a))
        )
        (progn
          (setq p5a (_vlax-curve-getpointatdist L3 (- d3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p5))))))
          (setq p7a (_vlax-curve-getpointatdist L3 (- d3 (+ (* k (/ d3 n_seg)) (* m1 (distance L1a p7))))))
          (if (null p5a) (setq p5a L3b))
          (if (null p7a) (setq p7a L3b))
        )
      )
      (setq ang1 (ang_der L3 p5a))
      (setq ang2 (ang_der L3 p7a))
      (setq p5b (polar p5a ang1 0.01))
      (setq p7b (polar p7a ang2 0.01))
      (setq L5 (entmakex (list (cons 0 "LINE") (cons 10 p5a) (cons 11 p5b))))
      (setq L6 (entmakex (list (cons 0 "LINE") (cons 10 p7a) (cons 11 p7b))))
      (setq L4# (vlax-ename->vla-object L4))
      (setq L5# (vlax-ename->vla-object L5))
      (setq L6# (vlax-ename->vla-object L6))
      (if (and
            (setq p6a (vlax-invoke L5# 'IntersectWith L4# acExtendThisEntity))
            (setq p8a (vlax-invoke L6# 'IntersectWith L4# acExtendThisEntity))
          )
        (progn
          (if (> (length p6a) 3) (setq p6a (p_vic p6a p5a)))
          (if (> (length p8a) 3) (setq p8a (p_vic p8a p7a)))
          (setq d (/ (* (distance p1 p5) (distance p5a p6a)) (distance p5 p6)))
          (setq p10 (polar p5a (angle p5a p6a) d))
          (setq d (/ (* (distance p2 p7) (distance p7a p8a)) (distance p7 p8)))
          (setq p11 (polar p7a (angle p7a p8a) d))
          (setq LineBlock (cons (list p10 p11) LineBlock))
        )
      )
      (entdel L5)
      (entdel L6)
      (setq i (1+ i))
      (if (= e_t 1) (acet-ui-progress-safe i))
    )
    (if (= e_t 1) (acet-ui-progress-done))
  )

  (defun ang_der (:a :b / DER :ang)
    (setq DER (vlax-curve-getFirstDeriv :a (vlax-curve-getParamAtPoint :a :b)))
    (cond
      ((/= (cadr DER) 0) (setq :ang (- pi (atan (/ (car DER) (cadr DER))))))
      ((= (cadr DER) 0) (setq :ang (/ pi 2)))
    )
    :ang
  )

  (defun p_vic (:j :k / :pp)
    (repeat (/ (length :j) 3)
      (setq lp (cons (list (car :j) (cadr :j) (caddr :j)) lp))
      (setq :j (cdddr :j))
    )
    (setq :pp
      (car
        (vl-sort lp
          (function
            (lambda (x1 x2)
              (<
                (distance :k (list (car x1) (cadr x1) (caddr x1)))
                (distance :k (list (car x2) (cadr x2) (caddr x2)))
              )
            )
          )
        )
      )
    )
    :pp
  )

  (defun draw_m3 (/ asse L_asse d5 n i dm1 dm2 dm3 dm4 p5a p6a p7a p8a p10 p11)
    (if command-s
      (command-s "_.offset" d_off L3 "_non" (_vlax-curve-getpointatdist L4 (/ d4 2.0)) "")
      (vl-cmdf "_.offset" d_off L3 "_non" (_vlax-curve-getpointatdist L4 (/ d4 2.0)) "")
    )
    (setq asse (entlast))
    (setq L_asse (vlax-curve-getDistAtParam asse (vlax-curve-getEndParam asse)))
    (setq d5 (distance Pm1 Pm2))
    (setq n (length Lin_B))
    (if (= e_t 1) (acet-ui-progress-init "" n))
    (setq i 0)
    (repeat n
      (punti_comuni Lin_B)
      (setq p12 (inters Pm1 Pm2 p1 p3 nil))
      (setq p13 (inters Pm1 Pm2 p2 p4 nil))
      (setq dm1 (distance Pm1 p12))
      (setq dm2 (distance Pm1 p13))
      (setq dm3 (/ (* dm1 (* (/ 1.0 n_seg) L_asse)) d5))
      (setq dm4 (/ (* dm2 (* (/ 1.0 n_seg) L_asse)) d5))
      (if (= L3rev 0)
        (progn
          (if (not (setq pm3 (_vlax-curve-getpointatdist asse (+ (* k (/ L_asse n_seg)) dm3))))
            (setq pm3 (_vlax-curve-getpointatdist asse L_asse))
          )
          (if (not (setq pm4 (_vlax-curve-getpointatdist asse (+ (* k (/ L_asse n_seg)) dm4))))
            (setq pm4 (_vlax-curve-getpointatdist asse L_asse))
          )
        )
        (progn
          (if (not (setq pm3 (_vlax-curve-getpointatdist asse (- L_asse (+ (* k (/ L_asse n_seg)) dm3)))))
            (setq pm3 (_vlax-curve-getpointatdist asse (abs (- L_asse (+ (* k (/ L_asse n_seg)) dm3)))))
          )
          (if (not (setq pm4 (_vlax-curve-getpointatdist asse (- L_asse (+ (* k (/ L_asse n_seg)) dm4)))))
            (setq pm4 (_vlax-curve-getpointatdist asse (abs (- L_asse (+ (* k (/ L_asse n_seg)) dm4)))))
          )
        )
      )
      (setq p5a (vlax-curve-getClosestPointTo L3 pm3))
      (setq p7a (vlax-curve-getClosestPointTo L3 pm4))
      (setq p6a (vlax-curve-getClosestPointTo L4 pm3))
      (setq p8a (vlax-curve-getClosestPointTo L4 pm4))
      (setq d (/ (* (distance p1 p5) (distance p5a p6a)) (distance p5 p6)))
      (setq p10 (polar p5a (angle p5a p6a) d))
      (setq d (/ (* (distance p2 p7) (distance p7a p8a)) (distance p7 p8)))
      (setq p11 (polar p7a (angle p7a p8a) d))
      (setq LineBlock (cons (list p10 p11) LineBlock))
      (setq i (1+ i))
      (if (= e_t 1) (acet-ui-progress-safe i))
    )
    (entdel asse)
    (if (= e_t 1) (acet-ui-progress-done))
  )

  (defun crea_blocco (/ str seed block_block modulus multiplier increment #)
    (setq str "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789")
    (setq seed (getvar "DATE"))
    (while
      (progn
        (setq block_block "")
        (repeat 7
          (setq modulus 65536
                multiplier 25173
                increment 13849
                seed (rem (+ (* multiplier seed) increment) modulus)
                block_block (strcat (substr str (1+ (fix (* 36 (/ seed modulus)))) 1) block_block)
          )
        )
        (tblsearch "BLOCK" (setq block_block (strcat "ALHM_" block_block)))
      )
    )
    (entmake (list '(0 . "BLOCK")
                   (cons 2 block_block) '(70 . 0)
                   (cons 10 L3a)
             )
    )
    (mapcar
      (function 
        (lambda (#)
          (entmake
            (list
              (cons 0 "LINE")
              (cons 8 b_lay)
              (cons 10 (car #))
              (cons 11 (cadr #))
              (cons 62 b_col)
            )
          )
        )
      )
      LineBlock
    )
    (entmake '((0 . "ENDBLK")))
    (entmake
      (list
        (cons 0 "INSERT")
        (cons 2 block_block)
        (cons 8 b_lay)
        (cons 10 L3a)
      )
    )
  )

  (defun alb_dcl (/ DCLname dcl_open)
    (setq DCLname (strcat (getvar 'localrootprefix) "ALIGNH_V1_0.dcl"))
    (if (not (findfile DCLname)) (crea_dcl_alb))
    (if (= del_b nil) (setq del_b 0))      ; delete original block 0=N 1=Y
    (if (= align_m nil) (setq align_m 1))  ; alignment method
    (if (= POSIZ_DCL_ALB nil) (setq POSIZ_DCL_ALB (list -1 -1)))
    (setq dcl_id (load_dialog DCLname))
    (if (not (new_dialog "ALB" dcl_id "" POSIZ_DCL_ALB)) (exit))
    (ALB_VECTORS)
    (setq dcl_open 1)
    (set_tile "align_m" (rtos align_m 2 0))
    (action_tile "1" "(setq method 1)(ALB_VECTORS)")
    (action_tile "2" "(setq method 2)(ALB_VECTORS)")
    (action_tile "3" "(setq method 3)(ALB_VECTORS)")
    (if (= del_b 0)
      (set_tile "del_b" "0")
      (set_tile "del_b" "1")
    )
    (setq del_block (rtos del_b 2 0))
    (action_tile "del_b" "(setq del_block $value)")
    (action_tile "accept" "(setq POSIZ_DCL_ALB (done_dialog 1))")
    (action_tile "cancel" "(setq POSIZ_DCL_ALB (done_dialog 2))")
    (setq :done (start_dialog))
    (unload_dialog dcl_id)
    (vl-file-delete DCLname)
    (if (= :done 2) (exit))
    (if (= :done 1)
      (progn
        (setq align_m method)
        (setq del_b (atoi del_block))
      )
    )
  )

  (defun ALB_VECTORS ()
    (start_image "ALB")
    (if (= method nil) (setq method align_m))
    (ALB_clean)
    (if (null dcl_open)
      (cond
        ((= align_m 1) (ALB_m1))
        ((= align_m 2) (ALB_m2))
        ((= align_m 3) (ALB_m3))
      )
      (cond
        ((= method 1) (ALB_m1))
        ((= method 2) (ALB_m2))
        ((= method 3) (ALB_m3))
      )
    )
    (end_image)
  )

  (defun ALB_clean ()
    (mapcar 'vector_image
            (list 254 255 258 261 262 264 265 267 269 269 268 267 265 259 253 248 245 241 233 227 223 220 308 303 305 314 315 317 319 321 322 323 322 321 319 317 316 313 309 305 300 288 281 277 274 272 266 264 262 261 307 307 307 308 309 310 312 315 318 319 321 322 325 327 330 332 333 334 335 336 335 334 332 331 330 328 323 270 273 276 279 281 281 281 280 277 273 268 263 257 254 252 252 253 254 253 252 252 252 253 254 255 257 261 273 275 277 279 280 281 280 279 278 276 275 271 270 448 446 445 441 439 438 435 432 432 431 427 427 426 422 422 423 418 418 421 416 415 419 414 414 419 413 413 419 413 413 419 414 414 421 416 416 429 422 415 409 404 400 397 395 394 394 395 397 400 404 409 416 423 424 418 419 427 422 423 431 427 428 436 432 433 442 439 440 466 460 457 454 453 450 448 446 445 444 443 443 443 444 445 447 450 455 457 464 466 469 425 422 413 410 408 405 404 400 399 396 395 394 393 393 393 394 395 396 397 399 400 402 405 408 410 413 419 420 425 296 280 280 279 280 282 287 292 297 302 306 308 309 308 306 303 299 290 291 294 294 297 298 300 300 302 302 303 303 303 302 301 300 298 297 293 293 288 288 283 282 278 278 275 274 274 274 274 275 275 672 672 673 673 674 674 674 674 675 675 675 675 676 676 676 676 676 676 676 676 676 675 675 675 675 674 674 674 674 673 673 672 672 671 671 670 670 670 670 669 669 669 669 668 668 668 668 668 668 668 668 668 668 668 669 669 669 669 670 670 670 670 671 671 672 673 673 674 675 675 675 675 675 674 673 673 672 672 670 670 669 669 669 669 669 669 670 670 670 671 672 672 673 673 673 673 674 674 674 674 674 673 673 673 673 672 672 671 671 671 671 670 670 670 670 670 670 670 671 671 671 671 672 672 673 673 673 672 672 671 671 671 671 671 672 674 674 675 675 674 674 671 671 670 669 669 669 671 672 672 673 673 674 674 674 675 675 676 676 676 676 677 677 677 677 677 677 677 677 677 676 676 676 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 668 668 668 667 667 667 667 667 667 667 667 667 667 667 668 668 668 668 669 669 670 670 670 671 671 506 506 506 507 507 508 508 508 508 509 509 509 509 510 510 510 510 510 510 510 510 510 509 509 509 509 508 508 508 508 507 507 506 506 505 505 504 504 504 504 503 503 503 503 502 502 502 502 502 502 502 502 502 502 502 503 503 503 503 504 504 504 504 505 505 507 507 508 508 509 509 509 509 509 509 508 508 508 507 506 506 505 504 503 503 503 503 503 503 503 504 504 505 506 506 506 507 507 507 507 508 508 508 508 508 507 507 507 507 506 506 505 505 505 505 504 504 504 504 504 504 504 505 505 505 505 506 506 507 507 507 506 506 505 505 505 505 505 506 508 509 507 505 504 504 503 503 504 505 506 506 507 507 508 508 508 509 509 510 510 510 510 511 511 511 511 511 511 511 511 511 510 510 510 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 502 502 502 501 501 501 501 501 501 501 501 501 501 501 502 502 502 502 503 503 504 504 504 505 505 441 441 441 442 442 443 443 443 443 444 444 444 444 445 445 445 445 445 445 445 445 445 444 444 444 444 443 443 443 443 442 442 441 441 440 440 439 439 439 439 438 438 438 438 437 437 437 437 437 437 437 437 437 437 437 438 438 438 438 439 439 439 439 440 440 441 441 442 443 443 444 444 444 444 444 444 443 443 442 442 441 440 439 438 438 438 438 438 438 438 439 439 440 440 441 441 442 442 442 442 443 443 443 443 443 442 442 442 442 441 441 440 440 440 440 439 439 439 439 439 439 439 440 440 440 440 441 441 442 442 442 441 441 440 440 440 440 440 442 443 444 443 441 440 439 439 438 438 439 441 441 442 442 443 443 443 444 444 445 445 445 445 446 446 446 446 446 446 446 446 446 445 445 445 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 437 437 437 436 436 436 436 436 436 436 436 436 436 436 437 437 437 437 438 438 439 439 439 440 440 261 261 261 262 262 263 263 263 263 264 264 264 264 265 265 265 265 265 265 265 264 264 264 264 263 263 263 263 262 262 261 261 260 260 259 259 259 259 258 258 258 258 257 257 257 257 257 257 257 257 257 257 257 258 258 258 258 259 259 259 259 260 260 261 262 263 263 264 264 264 264 264 263 262 261 261 260 260 259 259 259 258 258 258 258 258 258 259 259 259 260 261 261 262 262 262 262 263 263 263 263 263 262 262 262 262 261 261 260 260 259 259 259 259 259 259 259 260 260 261 261 262 262 262 261 261 260 260 260 260 260 260 260 261 262 263 262 258 258 258 260 261 261 262 262 263 263 263 264 264 265 265 265 265 266 266 266 266 266 266 266 266 266 265 265 265 265 264 264 264 263 263 263 262 262 261 261 260 260 259 259 259 258 258 257 257 257 256 256 256 256 256 256 256 256 256 256 256 257 257 257 257 258 258 259 259 259 260 260 173 173 173 174 174 175 175 175 175 176 176 176 176 177 177 177 177 177 177 177 177 177 176 176 176 176 175 175 175 175 174 174 173 173 172 172 171 171 171 171 170 170 170 170 169 169 169 169 169 169 169 169 169 169 169 170 170 170 170 171 171 171 171 172 172 173 174 174 175 175 176 176 176 176 176 176 176 175 175 174 174 173 172 172 171 171 171 170 170 170 170 170 170 170 170 171 172 172 173 173 174 174 174 174 175 175 175 175 175 174 174 174 174 173 173 172 172 172 172 171 171 171 171 171 171 171 172 172 172 172 173 173 174 174 174 173 173 172 172 172 172 172 173 175 175 173 170 171 171 173 173 174 174 175 175 175 176 176 177 177 177 177 178 178 178 178 178 178 178 178 178 177 177 177 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 169 169 169 168 168 168 168 168 168 168 168 168 168 168 169 169 169 169 170 170 171 171 171 172 172 188 188 188 188 183 180 175 172 175 177 177 177 177 177 176 172 172 169 169 168 167 166 165 164 164 164 161 161 160 158 158 157 156 156 156 156 156 157 158 158 160 161 161 184 187 177 178 179 180 181 182 182 178 177 177 177 177 178 179 180 181 172 169 172 174 174 174 174 174 173 169 169 166 159 159 159 617 614 612 610 608 607 606 606 606 606 607 608 609 611 613 616 619 511 507 505 502 500 499 498 498 498 498 499 501 503 505 508 512 618 616 615 613 611 609 608 607 606 605 605 605 606 607 608 609 611 612 616 617 618 620 510 508 507 505 503 501 500 499 498 497 497 497 498 499 500 501 503 504 508 509 510 512 672 564 265 262 261 261 261 262 266 268 271 276 280 288 298 304 308 316 322 322 321 319 318 316 310 309 218 216 216 216 217 218 219 220 222 224 228 232 241 244 250 257 260 264 266 269 270 268 267 262 260 254 398 395 393 391 391 391 392 393 394 396 402 404 407 411 418 430 435 439 443 446 447 449 451 452 453 452 451 449 447 445 444 438 435 433 349 348 347 346 345 345 346 347 349 350 353 357 363 365 371 375 378 383 389 395 397 398 399 400 399 397 395 394 392 391 388 385 384 216 215 214 216 220 227 236 246 255 262 267 269 268 265 261 257 258 259 263 263 266 267 269 269 270 270 269 269 265 265 259 258 250 249 241 239 231 230 223 223 218 218 216 216 216 216 217 218 433 396 393 392 393 398 406 417 427 438 447 453 454 452 448 444 438 422 423 428 428 433 434 437 438 441 441 442 442 442 441 438 438 432 430 422 421 412 410 401 400 391 389 383 382 380 380 382 382 385 385 387 391 395 398 400 400 400 400 400 398 392 382 369 355 346 346 348 259 263 266 269 270 269 265 258 249 240 231 223 218 216 216 217 254 259 263 266 270 269 265 258 249 240 231 223 218 216 216 217 163 163 163 148 148 148 163 541 538 537 536 535 535 535 536 537 539 541 544 546 548 550 552 554 557 560 564 566 567 568 567 566 565 563 559 557 554 549 547 544 494 493 492 491 490 489 488 488 488 489 490 492 494 497 511 513 515 517 519 520 521 520 519 517 514 511 506 502 502 508 514 519 521 520 517 513 508 502 497 492 489 488 488 490 492 664 668 672 674 676 677 679 680 681 680 679 678 676 674 672 668 666 665 664 663 620 618 616 615 614 611 610 609 608 607 606 605 604 604 604 605 606 607 608 609 610 612 614 615 617 618 620 621 617 614 611 609 607 606 605 604 604 605 606 608 610 612 615 618 399 391 384 378 373 369 366 364 363 363 365 367 370 374 379 385 392 436 430 427 424 422 420 418 416 415 414 413 413 413 414 415 417 420 425 427 434 436 439 395 392 383 380 378 375 374 370 369 366 365 364 363 363 363 364 365 366 367 369 370 372 375 378 380 383 389 390 395 50 54 53 51 50 48 47 45 44 42 41 39 38 37 35 34 32 32 41 105 107 108 110 111 112 114 115 117 118 120 121 123 124 126 127 123 114 105 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 166 166 166 151 151 151 166 166 165 166 165 164 164 150 150 165 165 150 164 164 149 149 149 261 263 265 268 217 216 215 215 216 217 218 219)
            (list 271 268 258 248 244 237 233 224 212 180 175 171 165 153 144 137 133 128 117 108 101 94 265 279 273 247 244 237 230 220 214 192 185 180 174 169 167 162 156 151 145 132 124 119 115 112 102 97 88 80 61 61 73 84 89 93 100 108 116 119 124 127 135 141 151 159 164 169 175 188 220 228 240 245 250 259 279 265 251 238 225 212 200 188 176 164 152 139 125 109 91 74 57 41 34 44 61 61 73 88 94 100 108 121 153 159 166 174 179 189 217 224 230 239 244 260 265 261 270 268 249 258 256 237 245 243 224 232 230 211 219 217 198 205 203 185 192 190 171 178 176 158 164 162 144 150 148 139 136 134 126 122 120 279 266 252 238 224 209 194 179 163 148 133 118 103 88 73 59 46 112 108 106 99 95 93 86 81 79 74 68 67 61 56 54 69 80 87 95 100 107 114 124 130 137 150 150 162 175 184 194 206 221 226 240 244 249 40 45 63 70 75 83 86 100 104 119 126 134 148 148 163 178 187 194 200 209 213 220 230 238 243 250 262 264 273 275 46 53 68 82 97 111 125 138 152 166 181 195 210 232 246 261 272 270 257 256 243 241 229 227 215 213 201 199 187 185 173 171 160 158 146 144 132 130 119 117 104 102 87 74 72 59 57 44 42 220 220 220 220 220 219 219 219 219 218 218 218 218 217 217 216 216 215 215 214 214 214 214 213 213 213 213 212 212 212 212 212 212 212 212 212 212 212 213 213 213 213 214 214 214 214 215 215 216 216 217 217 218 218 218 218 219 219 219 219 220 220 220 220 219 219 219 218 217 217 216 216 215 214 213 213 213 213 213 214 214 215 216 217 217 218 218 218 219 219 218 218 218 217 217 217 217 216 216 215 215 215 215 214 214 214 214 214 214 214 215 215 215 215 216 216 217 217 217 217 218 218 217 217 216 216 215 215 215 215 215 216 216 217 219 218 218 215 214 214 213 213 213 213 214 215 216 219 221 221 221 221 221 220 220 220 220 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 212 212 212 211 211 211 211 211 211 211 211 211 211 211 212 212 213 213 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 220 220 220 220 221 221 221 221 231 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 235 235 234 234 233 233 232 231 231 230 230 230 229 229 229 229 229 230 230 230 231 232 233 233 234 234 235 235 235 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 233 233 232 232 231 231 231 231 231 232 232 233 235 234 232 229 229 229 229 231 232 234 235 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 227 227 228 228 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 255 260 260 260 260 260 259 259 259 259 258 258 258 258 257 257 256 256 255 255 254 254 254 254 253 253 253 253 252 252 252 252 252 252 252 252 252 252 252 253 253 253 253 254 254 254 254 255 255 256 256 257 257 258 258 258 258 259 259 259 259 260 260 260 260 259 259 259 258 258 257 257 256 255 255 254 254 254 253 253 253 253 254 254 255 255 256 257 257 258 258 259 259 259 258 258 258 257 257 257 257 256 256 255 255 255 255 254 254 254 254 254 254 254 255 255 255 255 256 256 257 257 257 257 258 258 257 257 256 256 255 255 255 255 255 256 256 257 259 258 256 253 253 253 253 253 254 256 258 261 261 261 261 261 260 260 260 260 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 252 252 252 251 251 251 251 251 251 251 251 251 251 251 252 252 253 253 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 260 260 260 260 261 261 261 261 247 252 252 252 252 252 251 251 251 251 250 250 250 249 249 248 248 247 247 246 246 246 245 245 245 245 244 244 244 244 244 244 244 244 244 244 244 245 245 245 245 246 246 246 246 247 247 248 248 249 249 250 250 250 250 251 251 251 251 252 252 252 252 251 251 250 250 249 248 248 247 247 246 245 245 245 245 245 245 245 246 246 247 248 248 249 250 250 250 251 251 250 250 250 249 249 249 249 248 248 247 247 247 247 246 246 246 246 246 246 247 247 247 248 248 249 249 250 250 249 249 248 248 247 247 247 247 247 247 248 248 249 249 251 251 246 245 246 247 249 251 253 253 253 253 253 252 252 252 252 251 250 250 250 250 249 249 248 248 247 247 246 246 246 246 245 245 245 244 244 244 243 243 243 243 243 243 243 243 243 243 243 244 244 244 244 245 246 246 246 247 247 248 248 249 249 250 250 251 251 252 252 252 252 252 253 253 253 253 231 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 235 235 235 234 234 233 233 232 232 231 231 230 230 230 229 229 229 229 229 229 229 230 230 231 231 232 232 233 233 234 234 235 235 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 233 233 232 232 231 231 231 231 231 232 232 233 235 234 229 229 230 234 235 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 227 227 228 228 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 264 260 256 256 264 256 260 260 260 259 258 258 257 256 256 256 256 256 262 264 264 264 264 263 262 256 263 263 264 264 264 263 262 261 259 258 258 257 256 256 256 256 257 244 244 251 252 252 252 252 250 249 247 247 246 245 245 244 244 244 244 248 248 248 247 246 246 245 244 244 244 244 244 248 244 244 42 55 67 80 93 106 119 132 145 158 171 184 197 210 223 236 249 35 49 64 78 93 107 122 137 151 166 181 195 210 224 238 253 248 239 235 225 213 199 191 181 167 148 130 130 115 103 93 85 71 65 45 41 36 28 248 239 235 225 213 199 191 181 167 148 130 130 115 103 93 85 71 65 45 41 36 28 42 42 47 63 73 73 79 89 102 106 111 118 123 132 143 150 155 167 185 215 221 230 234 241 259 262 36 56 56 77 83 88 91 94 99 103 110 116 128 132 140 150 155 163 168 180 188 219 224 244 251 271 35 47 57 74 74 80 88 93 97 102 112 115 119 124 132 145 151 156 162 167 169 174 180 185 192 214 220 230 237 244 247 265 273 279 29 36 44 51 63 63 76 83 91 94 101 108 117 120 128 133 137 144 153 165 171 175 180 189 212 224 233 237 244 248 258 268 271 43 51 66 82 97 111 123 136 148 161 176 191 214 229 244 259 257 255 243 241 228 226 212 210 197 195 181 179 167 165 153 151 140 138 128 126 115 113 102 100 87 85 72 70 57 55 41 39 282 49 65 73 88 103 117 129 141 153 166 181 205 221 237 252 267 276 274 261 259 246 245 232 230 217 215 203 201 189 187 176 175 165 163 153 151 141 140 129 127 115 113 97 95 79 77 61 59 45 43 262 247 233 220 209 200 195 192 186 173 158 143 126 104 78 55 35 256 242 227 211 196 180 166 152 139 127 114 101 86 71 56 40 271 256 241 226 196 180 165 152 139 127 114 101 86 71 56 40 117 169 155 155 131 131 117 37 52 57 63 73 73 82 91 95 102 108 116 121 126 131 136 141 149 158 172 181 188 197 224 230 235 242 254 259 266 277 281 286 32 37 42 47 53 60 71 71 83 95 100 109 116 125 160 166 172 178 186 191 198 220 225 232 240 247 258 266 266 253 239 225 210 196 181 167 153 140 126 112 97 83 68 53 39 60 72 86 94 104 110 124 135 152 178 187 195 207 217 225 239 245 248 251 254 42 49 58 62 67 82 88 95 102 110 121 135 150 150 165 182 197 208 216 223 230 242 252 256 265 269 276 280 266 252 237 223 208 193 179 164 149 135 120 106 91 77 62 48 279 266 252 238 224 209 194 179 163 148 133 118 103 88 73 59 46 69 80 87 95 100 107 114 124 130 137 150 150 162 175 184 194 206 221 226 240 244 249 40 45 63 70 75 83 86 100 104 119 126 134 148 148 163 178 187 194 200 209 213 220 230 238 243 250 262 264 273 264 40 47 70 77 99 114 129 144 158 173 188 203 209 232 239 264 264 264 30 47 62 77 91 106 121 136 150 165 180 195 209 224 239 254 264 264 30 30 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44 162 124 124 134 134 152 152 123 121 163 165 119 167 133 153 153 121 133 154 119 132 154 132 74 57 47 35 83 76 63 63 51 44 36 29)
            (list 255 258 261 262 264 265 267 269 270 270 269 268 267 265 259 253 248 245 235 233 227 223 316 305 308 315 317 319 321 322 323 323 323 322 321 319 317 316 313 309 305 300 288 281 277 274 272 266 263 262 309 307 308 309 310 312 315 318 319 321 322 325 327 330 332 333 334 335 336 336 336 335 334 332 331 330 328 323 327 330 333 335 336 336 334 330 325 319 314 310 308 307 307 308 255 254 253 252 253 254 255 257 261 273 275 277 279 280 281 281 281 280 279 278 276 275 271 453 456 455 445 449 448 439 443 442 434 439 438 429 434 433 425 430 430 422 428 427 420 426 426 419 425 425 419 425 425 420 426 426 423 427 428 472 465 460 455 452 448 446 445 444 444 445 446 449 452 456 461 466 426 430 431 430 434 434 435 438 439 440 443 444 447 449 450 472 466 460 457 454 453 450 448 446 445 444 443 444 445 447 450 455 457 464 466 469 471 430 425 422 413 410 408 405 404 400 399 396 395 394 393 394 395 396 397 399 400 402 405 408 410 413 419 420 425 429 298 281 280 280 281 285 289 294 299 304 307 309 309 308 307 304 301 302 302 305 306 309 309 312 312 314 314 315 315 315 314 313 312 309 309 305 304 299 299 294 294 290 289 287 286 286 286 286 287 287 672 673 673 674 674 674 674 675 675 675 675 676 676 676 676 676 676 676 676 676 676 676 675 675 675 675 674 674 674 674 673 673 672 672 671 671 670 670 670 670 669 669 669 669 668 668 668 668 668 668 668 668 668 669 669 669 669 670 670 670 670 671 671 672 672 673 674 674 675 675 675 675 675 674 674 673 673 672 670 670 670 669 669 669 669 670 670 670 671 671 672 673 673 673 673 674 674 674 674 674 674 674 673 673 673 673 672 672 671 671 671 671 670 670 670 670 670 671 671 671 671 672 672 673 673 673 673 673 672 672 671 671 671 672 673 674 675 675 675 675 674 672 671 671 669 669 669 672 672 673 673 674 674 674 675 675 676 676 676 676 677 677 677 677 677 677 677 677 677 677 677 676 676 676 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 668 668 668 667 667 667 667 667 667 667 667 667 668 668 668 668 669 669 670 670 670 671 671 672 506 506 507 507 508 508 508 508 509 509 509 509 510 510 510 510 510 510 510 510 510 510 510 509 509 509 509 508 508 508 508 507 507 506 506 505 505 504 504 504 504 503 503 503 503 502 502 502 502 502 502 502 502 502 503 503 503 503 504 504 504 504 505 505 506 507 508 508 508 509 509 509 509 509 509 509 508 508 507 507 506 505 504 504 503 503 503 503 503 504 504 505 505 506 506 507 507 507 507 508 508 508 508 508 508 508 507 507 507 507 506 506 505 505 505 505 504 504 504 504 504 505 505 505 505 506 506 507 507 507 507 507 506 506 505 505 505 506 507 509 509 508 506 505 504 503 503 504 506 506 507 507 508 508 508 509 509 510 510 510 510 511 511 511 511 511 511 511 511 511 511 511 510 510 510 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 502 502 502 501 501 501 501 501 501 501 501 501 502 502 502 502 503 503 504 504 504 505 505 506 441 441 442 442 443 443 443 443 444 444 444 444 445 445 445 445 445 445 445 445 445 445 445 444 444 444 444 443 443 443 443 442 442 441 441 440 440 439 439 439 439 438 438 438 438 437 437 437 437 437 437 437 437 437 438 438 438 438 439 439 439 439 440 440 441 441 442 442 443 444 444 444 444 444 444 444 444 443 443 442 441 440 439 438 438 438 438 438 438 439 439 440 440 441 441 442 442 442 442 443 443 443 443 443 443 443 442 442 442 442 441 441 440 440 440 440 439 439 439 439 439 440 440 440 440 441 441 442 442 442 442 442 441 441 440 440 440 441 443 443 444 443 442 441 440 439 439 438 439 441 442 442 443 443 443 444 444 445 445 445 445 446 446 446 446 446 446 446 446 446 446 446 445 445 445 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 437 437 437 436 436 436 436 436 436 436 436 436 437 437 437 437 438 438 439 439 439 440 440 441 261 261 262 262 263 263 263 263 264 264 264 264 265 265 265 265 265 265 265 265 265 264 264 264 264 263 263 263 263 262 262 261 261 260 260 259 259 259 259 258 258 258 258 257 257 257 257 257 257 257 257 257 258 258 258 258 259 259 259 259 260 260 261 261 262 263 264 264 264 264 264 264 263 262 262 261 261 260 260 259 259 259 258 258 258 258 259 259 259 260 260 261 262 262 262 262 263 263 263 263 263 263 263 262 262 262 262 261 261 260 260 259 259 259 259 259 260 260 261 261 262 262 262 262 262 261 261 260 260 260 260 260 261 262 263 264 263 258 258 258 261 261 262 262 263 263 263 264 264 265 265 265 265 266 266 266 266 266 266 266 266 266 266 266 265 265 265 265 264 264 264 263 263 263 262 262 261 261 260 260 259 259 259 258 258 257 257 257 256 256 256 256 256 256 256 256 256 257 257 257 257 258 258 259 259 259 260 260 261 173 173 174 174 175 175 175 175 176 176 176 176 177 177 177 177 177 177 177 177 177 177 177 176 176 176 176 175 175 175 175 174 174 173 173 172 172 171 171 171 171 170 170 170 170 169 169 169 169 169 169 169 169 169 170 170 170 170 171 171 171 171 172 172 173 173 174 175 175 176 176 176 176 176 176 176 176 176 175 175 174 173 173 172 172 171 171 171 170 170 170 170 170 170 171 171 172 173 173 174 174 174 174 175 175 175 175 175 175 175 174 174 174 174 173 173 172 172 172 172 171 171 171 171 171 172 172 172 172 173 173 174 174 174 174 174 173 173 172 172 172 173 174 175 175 174 170 171 172 173 174 174 175 175 175 176 176 177 177 177 177 178 178 178 178 178 178 178 178 178 178 178 177 177 177 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 169 169 169 168 168 168 168 168 168 168 168 168 169 169 169 169 170 170 171 171 171 172 172 173 193 191 193 188 186 183 177 175 177 177 177 177 177 177 177 176 172 169 169 169 168 167 166 165 164 164 162 161 161 160 158 158 157 156 156 156 157 158 158 160 161 161 162 190 187 178 179 180 181 182 182 182 182 178 177 177 178 179 180 181 182 174 172 174 174 174 174 174 174 174 173 169 166 162 164 159 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 564 564 564 564 564 564 564 564 564 564 564 564 564 564 564 564 620 618 616 615 613 611 609 608 607 606 605 606 607 608 609 611 612 616 617 618 620 622 514 510 508 507 505 503 501 500 499 498 497 498 499 500 501 503 504 508 509 510 512 514 672 564 269 263 262 261 262 264 268 271 276 280 288 298 304 308 311 319 323 323 322 321 319 318 314 310 220 217 216 217 218 220 220 222 224 228 232 241 244 250 253 260 264 266 268 270 270 269 268 265 262 260 399 398 395 393 391 392 393 394 396 402 404 407 411 418 430 435 439 443 446 447 449 451 452 453 453 453 452 451 449 447 445 444 438 435 350 349 348 347 346 346 347 349 350 353 357 363 365 371 375 378 383 389 395 397 398 399 400 400 400 399 397 395 394 392 391 388 385 217 215 215 217 223 232 241 250 259 265 269 269 269 266 263 259 269 269 273 274 277 277 280 280 281 281 280 280 276 275 268 267 259 258 249 248 241 239 233 232 229 228 227 227 227 227 228 229 436 398 394 392 395 402 411 422 433 443 450 454 455 454 450 446 441 433 433 438 439 443 444 448 449 452 452 453 453 453 452 448 448 441 439 431 429 420 419 409 408 400 399 394 393 391 391 392 393 395 396 433 439 444 448 452 453 452 448 440 430 419 409 399 393 391 393 396 306 312 317 321 323 318 304 289 277 268 263 261 261 261 263 265 303 308 313 318 323 322 318 310 300 289 278 269 263 261 262 265 189 189 163 163 148 163 163 542 541 538 537 536 535 536 537 539 541 544 546 548 550 552 554 557 560 564 566 567 568 568 568 567 566 565 563 559 557 554 549 547 496 494 493 492 491 490 489 488 489 490 492 494 497 511 513 515 517 519 520 521 521 521 520 519 517 514 511 506 544 551 557 562 566 568 568 566 562 558 552 547 541 537 535 536 538 668 672 674 676 677 679 680 681 681 681 680 679 678 676 674 672 668 666 665 664 623 620 618 616 615 614 611 610 609 608 607 606 605 604 605 606 607 608 609 610 612 614 615 617 618 620 621 663 667 670 673 676 678 679 680 681 681 680 679 678 676 674 671 668 441 435 430 425 421 418 416 414 413 413 414 416 418 422 426 430 436 442 436 430 427 424 422 420 418 416 415 414 413 414 415 417 420 425 427 434 436 439 441 400 395 392 383 380 378 375 374 370 369 366 365 364 363 364 365 366 367 369 370 372 375 378 380 383 389 390 395 399 110 55 53 52 50 49 48 46 45 43 42 40 39 37 36 34 33 37 46 106 107 109 110 112 113 115 116 118 119 121 122 123 125 126 128 128 119 105 55 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 185 185 166 166 151 166 166 186 187 186 187 188 188 150 165 165 165 165 164 164 149 164 164 263 265 268 269 219 217 216 216 217 218 219 220)
            (list 268 258 248 244 237 233 224 212 202 189 180 175 171 165 153 144 137 133 120 117 108 101 241 273 265 244 237 230 220 214 203 206 192 185 180 174 169 167 162 156 151 145 132 124 119 115 112 102 93 88 31 73 84 89 93 100 108 116 119 124 127 135 141 151 159 164 169 175 188 203 203 220 228 240 245 250 259 279 265 250 234 218 201 183 166 149 134 119 106 94 83 72 59 46 26 34 44 73 88 94 100 108 121 153 159 166 174 179 189 205 205 217 224 230 239 244 260 269 263 261 257 252 250 245 240 238 233 228 226 220 215 213 207 203 201 194 190 188 180 177 175 167 163 161 153 150 148 130 137 135 117 124 122 252 242 231 220 209 197 186 174 162 150 138 126 114 103 92 81 70 104 111 109 91 98 96 78 86 84 66 74 72 54 62 60 59 69 80 87 95 100 107 114 124 130 137 162 175 184 194 206 221 226 240 244 249 252 32 40 45 63 70 75 83 86 100 104 119 126 134 163 178 187 194 200 209 213 220 230 238 243 250 262 264 273 279 267 39 61 75 90 104 118 132 145 159 174 188 203 217 224 239 253 275 273 260 258 246 244 231 229 216 214 201 199 186 184 171 169 156 154 142 140 128 126 114 113 101 99 85 74 72 59 57 45 43 220 220 220 220 220 220 219 219 219 219 218 218 218 218 217 217 216 216 215 215 214 214 214 214 213 213 213 213 212 212 212 212 212 212 212 212 212 213 213 213 213 214 214 214 214 215 215 216 216 217 217 218 218 218 218 219 219 219 219 220 220 220 220 220 219 219 219 218 218 217 217 216 215 214 213 213 213 213 214 214 214 215 216 217 218 218 218 219 219 219 218 218 218 218 217 217 217 217 216 216 215 215 215 215 214 214 214 214 214 215 215 215 215 216 216 217 217 217 217 218 218 218 217 217 217 216 216 215 215 215 216 216 217 217 219 219 218 216 215 214 214 213 213 213 215 216 217 219 221 221 221 221 221 221 220 220 220 220 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 212 212 212 211 211 211 211 211 211 211 211 211 212 212 212 212 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 220 220 220 220 221 221 221 221 221 232 236 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 236 235 235 235 234 234 233 232 232 231 231 230 230 230 229 229 229 229 230 230 231 231 232 233 234 234 234 235 235 235 234 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 234 233 233 233 232 232 231 231 231 232 232 233 233 235 234 233 229 229 229 230 232 233 235 235 237 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 228 228 228 228 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 237 256 260 260 260 260 260 260 259 259 259 259 258 258 258 258 257 257 256 256 255 255 254 254 254 254 253 253 253 253 252 252 252 252 252 252 252 252 252 253 253 253 253 254 254 254 254 255 255 256 256 257 257 258 258 258 258 259 259 259 259 260 260 260 260 260 259 259 259 258 258 258 257 256 256 255 255 254 254 253 253 253 253 254 255 255 256 256 257 258 258 258 259 259 259 258 258 258 258 257 257 257 257 256 256 255 255 255 255 254 254 254 254 254 255 255 255 255 256 256 257 257 257 257 258 258 258 257 257 257 256 256 255 255 255 256 256 257 257 259 259 257 254 253 253 253 254 254 257 259 261 261 261 261 261 261 260 260 260 260 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 252 252 252 251 251 251 251 251 251 251 251 251 252 252 252 252 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 260 260 260 260 261 261 261 261 261 248 252 252 252 252 252 252 251 251 251 251 250 250 250 249 249 248 248 247 247 246 246 246 245 245 245 245 244 244 244 244 244 244 244 244 244 245 245 245 245 246 246 246 246 247 247 248 248 249 249 250 250 250 250 251 251 251 251 252 252 252 252 252 251 251 250 249 249 249 248 248 247 246 245 245 245 245 245 245 246 246 246 247 248 249 249 250 250 251 251 251 250 250 250 250 249 249 249 249 248 248 247 247 247 247 246 246 246 246 246 246 247 248 248 249 249 250 250 250 249 249 249 248 248 247 247 247 247 248 248 249 249 249 251 250 247 246 247 248 250 251 253 253 253 253 253 253 252 252 251 251 251 250 250 250 250 249 249 248 248 247 247 246 246 246 246 245 245 245 244 244 244 243 243 243 243 243 243 243 243 243 244 244 244 244 245 245 245 246 247 247 248 248 249 249 250 250 251 251 252 252 252 252 252 253 253 253 253 253 232 236 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 236 235 235 235 234 234 234 233 233 232 232 231 231 230 230 229 229 229 229 229 229 230 230 230 231 232 232 233 233 234 234 234 235 235 234 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 234 233 233 233 232 232 231 231 231 232 232 233 233 235 235 230 229 231 235 235 237 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 228 228 228 228 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 237 264 260 256 264 256 264 264 260 259 259 259 258 258 257 256 256 264 262 263 263 264 264 264 264 263 262 262 264 264 264 264 264 263 262 261 259 257 256 256 256 256 257 258 244 252 252 252 252 252 251 251 250 249 247 247 246 244 244 244 244 245 252 248 247 247 247 246 246 245 244 244 252 252 248 244 252 42 55 67 80 93 106 119 132 145 158 171 184 197 210 223 236 249 48 61 74 87 100 112 125 137 150 162 175 188 200 213 226 239 256 248 239 235 225 213 199 191 181 167 148 115 103 93 85 71 65 45 41 36 28 21 264 248 239 235 225 213 199 191 181 167 148 115 103 93 85 71 65 45 41 36 28 21 262 262 30 57 63 80 89 97 106 111 118 123 132 143 150 155 159 174 193 206 215 221 230 234 247 259 24 44 77 83 88 94 94 99 103 110 116 128 132 140 144 155 163 168 175 188 202 212 219 233 244 251 30 35 47 57 80 88 93 97 102 112 115 119 124 132 145 151 156 162 167 169 174 180 185 192 203 203 214 220 230 237 244 247 265 273 24 29 36 44 51 76 83 91 94 101 108 117 120 128 133 137 144 153 165 171 175 180 189 202 202 212 224 233 237 244 248 258 268 35 59 74 90 104 117 130 142 155 169 184 199 206 221 236 251 261 259 245 243 230 228 214 212 197 195 180 178 163 161 147 145 133 132 121 119 109 107 97 95 85 83 72 70 57 55 43 41 275 41 57 81 96 110 123 135 147 159 173 189 197 213 229 244 260 280 278 265 263 250 248 235 233 219 217 203 201 187 185 172 170 158 156 146 144 134 132 122 120 109 107 94 92 79 77 63 61 47 45 279 264 249 234 218 202 186 171 157 145 133 121 108 93 78 62 46 271 255 237 218 196 173 150 133 119 106 93 82 76 69 59 47 279 264 249 234 202 186 171 157 145 133 121 108 93 77 62 46 143 143 169 155 155 131 131 33 37 52 57 63 82 91 95 102 108 116 121 126 131 136 141 149 158 172 181 188 197 213 213 224 230 235 242 254 259 266 277 281 23 32 37 42 47 53 60 83 95 100 109 116 125 160 166 172 178 186 191 198 211 211 220 225 232 240 247 258 287 273 259 244 229 214 198 183 168 153 139 124 110 95 80 65 49 72 86 94 104 110 124 135 152 165 165 178 187 195 207 217 225 239 245 248 251 33 42 49 58 62 67 82 88 95 102 110 121 135 165 182 197 208 216 223 230 242 252 256 265 269 276 280 254 242 231 220 209 197 186 174 163 151 140 128 117 105 94 83 72 252 242 231 220 209 197 186 174 162 150 138 126 114 103 92 81 70 59 69 80 87 95 100 107 114 124 130 137 162 175 184 194 206 221 226 240 244 249 252 32 40 45 63 70 75 83 86 100 104 119 126 134 163 178 187 194 200 209 213 220 230 238 243 250 262 264 273 279 264 30 55 62 85 91 106 121 136 150 165 180 195 217 224 247 254 264 264 40 55 70 85 99 114 129 144 158 173 188 203 217 232 247 264 264 264 264 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44 143 143 134 134 152 152 162 143 143 143 143 143 143 153 153 165 133 133 167 132 154 154 132 57 47 35 30 91 83 76 51 44 36 29 24)
            (list -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15 -15)
    )
  )

  (defun ALB_m1 ()
    (mapcar 'vector_image
            (list 50 54 53 51 50 48 47 45 44 42 41 39 38 37 35 34 32 32 41 105 107 108 110 111 112 114 115 117 118 120 121 123 124 126 127 123 114 105)
            (list 264 40 47 70 77 99 114 129 144 158 173 188 203 209 232 239 264 264 264 30 47 62 77 91 106 121 136 150 165 180 195 209 224 239 254 264 264 264)
            (list 55 55 53 52 50 49 48 46 45 43 42 40 39 37 36 34 33 37 46 106 107 109 110 112 113 115 116 118 119 121 122 123 125 126 128 128 119 110)
            (list 264 30 55 62 85 91 106 121 136 150 165 180 195 217 224 247 254 264 264 40 55 70 85 99 114 129 144 158 173 188 203 217 232 247 264 264 264 264)
            (list 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2)
    )
    (mapcar 'vector_image
            (list 254 259 263 266 269 270 269 265 258 249 240 231 223 218 216 216 217 502 508 514 519 521 520 517 513 508 502 497 492 489 488 488 490 492 621 617 614 611 609 607 606 605 604 604 605 606 608 610 612 615 618 399 391 384 378 373 369 366 364 363 363 365 367 370 374 379 385 392 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55)
            (list 271 256 241 226 211 196 180 165 152 139 127 114 101 86 71 56 40 266 253 239 225 210 196 181 167 153 140 126 112 97 83 68 53 39 280 266 252 237 223 208 193 179 164 149 135 120 106 91 77 62 48 279 266 252 238 224 209 194 179 163 148 133 118 103 88 73 59 46 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44)
            (list 303 308 313 318 321 323 322 318 310 300 289 278 269 263 261 262 265 544 551 557 562 566 568 568 566 562 558 552 547 541 537 535 536 538 663 667 670 673 676 678 679 680 681 681 680 679 678 676 674 671 668 441 435 430 425 421 418 416 414 413 413 414 416 418 422 426 430 436 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105)
            (list 279 264 249 234 218 202 186 171 157 145 133 121 108 93 77 62 46 287 273 259 244 229 214 198 183 168 153 139 124 110 95 80 65 49 254 242 231 220 209 197 186 174 163 151 140 128 117 105 94 83 72 252 242 231 220 209 197 186 174 162 150 138 126 114 103 92 81 70 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44)
            (list 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1)
    )
    (mapcar 'vector_image
            (list 541 540 539 538 537 536 535 535 535 536 537 539 541 544 546 548 550 552 554 557 559 560 562 564 566 567 568 567 566 565 563 561 559 557 554 549 547 544 494 493 492 491 490 489 488 488 488 489 490 492 494 497 499 501 503 505 507 509 511 513 515 517 518 519 520 521 520 519 517 514 511 506 502 664 666 668 670 672 673 674 675 676 677 678 679 680 681 680 679 678 677 676 675 674 673 672 670 668 666 665 664 663 620 618 616 615 614 612 611 610 609 608 607 606 605 604 604 604 605 606 607 608 609 610 611 612 614 615 617 618 620 439 436 430 427 424 422 420 418 417 416 415 414 413 413 413 414 415 416 417 418 419 420 421 423 425 427 429 430 434 436 439 395 392 389 385 383 380 378 375 374 372 370 369 368 367 366 365 364 363 363 363 364 365 366 367 369 370 372 375 378 380 383 385 389 390 395 105 55 303 305 308 310 311 314 315 317 319 321 322 323 322 321 319 317 316 313 311 309 305 300 288 281 277 274 272 266 264 263 262 261 261 261 263 264 265 266 268 254 255 258 261 262 264 265 267 268 269 270 269 268 267 266 265 264 259 253 248 245 241 238 235 233 227 223 220 219 218 217 216 215 215 216 217 218 219)
            (list 37 42 47 52 57 63 73 73 82 91 95 102 108 116 121 126 131 136 141 149 155 158 165 172 181 188 197 224 230 235 242 248 254 259 266 277 281 286 32 37 42 47 53 60 71 71 83 95 100 109 116 125 130 135 140 145 150 155 160 166 172 178 182 186 191 198 220 225 232 240 247 258 266 60 66 72 79 86 90 94 99 104 110 117 124 135 152 178 187 195 201 207 212 217 221 225 232 239 245 248 251 254 42 49 58 62 67 77 82 88 95 102 110 121 135 150 150 165 182 197 208 216 223 230 236 242 252 256 265 269 276 64 69 80 87 95 100 107 114 119 124 130 137 150 150 162 175 184 189 194 198 202 206 209 215 221 226 230 232 240 244 249 40 45 51 59 63 70 75 83 86 93 100 104 109 114 119 126 134 148 148 163 178 187 194 200 209 213 220 230 238 243 250 254 262 264 273 30 30 279 273 265 259 256 247 244 237 230 220 214 192 185 180 174 169 167 162 159 156 151 145 132 124 119 115 112 102 97 93 88 80 74 74 57 52 47 43 35 271 268 258 248 244 237 233 224 218 212 189 180 175 171 168 165 163 153 144 137 133 128 124 120 117 108 101 94 91 87 83 76 63 63 51 44 36 29)
            (list 542 541 540 539 538 537 536 535 536 537 539 541 544 546 548 550 552 554 557 559 560 562 564 566 567 568 568 568 567 566 565 563 561 559 557 554 549 547 496 494 493 492 491 490 489 488 489 490 492 494 497 499 501 503 505 507 509 511 513 515 517 518 519 520 521 521 521 520 519 517 514 511 506 666 668 670 672 673 674 675 676 677 678 679 680 681 681 681 680 679 678 677 676 675 674 673 672 670 668 666 665 664 623 620 618 616 615 614 612 611 610 609 608 607 606 605 604 605 606 607 608 609 610 611 612 614 615 617 618 620 621 442 439 436 430 427 424 422 420 418 417 416 415 414 413 414 415 416 417 418 419 420 421 423 425 427 429 430 434 436 439 441 400 395 392 389 385 383 380 378 375 374 372 370 369 368 367 366 365 364 363 364 365 366 367 369 370 372 375 378 380 383 385 389 390 395 399 105 55 305 308 310 311 314 315 317 319 321 322 323 323 323 322 321 319 317 316 313 311 309 305 300 288 281 277 274 272 266 264 263 262 261 263 264 265 266 268 269 255 258 261 262 264 265 267 268 269 270 270 270 269 268 267 266 265 264 259 253 248 245 241 238 235 233 227 223 220 219 218 217 216 216 217 218 219 220)
            (list 33 37 42 47 52 57 63 82 91 95 102 108 116 121 126 131 136 141 149 155 158 165 172 181 188 197 213 213 224 230 235 242 248 254 259 266 277 281 23 32 37 42 47 53 60 83 95 100 109 116 125 130 135 140 145 150 155 160 166 172 178 182 186 191 198 211 211 220 225 232 240 247 258 66 72 79 86 90 94 99 104 110 117 124 135 152 165 165 178 187 195 201 207 212 217 221 225 232 239 245 248 251 33 42 49 58 62 67 77 82 88 95 102 110 121 135 165 182 197 208 216 223 230 236 242 252 256 265 269 276 280 59 64 69 80 87 95 100 107 114 119 124 130 137 162 175 184 189 194 198 202 206 209 215 221 226 230 232 240 244 249 252 32 40 45 51 59 63 70 75 83 86 93 100 104 109 114 119 126 134 163 178 187 194 200 209 213 220 230 238 243 250 254 262 264 273 279 264 264 273 265 259 256 247 244 237 230 220 214 203 203 192 185 180 174 169 167 162 159 156 151 145 132 124 119 115 112 102 97 93 88 80 57 52 47 43 35 30 268 258 248 244 237 233 224 218 212 202 202 189 180 175 171 168 165 163 153 144 137 133 128 124 120 117 108 101 94 91 87 83 76 51 44 36 29 24)
            (list 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250)
    )
    (mapcar 'vector_image
            (list 163 163 163 148 148 148 163 166 166 166 151 151 151 166 166 165 166 165 164 164 150 150 165 165 150 164 164 149 149 149)
            (list 117 169 155 155 131 131 117 162 124 124 134 134 152 152 123 121 163 165 119 167 133 153 153 121 133 154 119 132 154 132)
            (list 189 189 163 163 148 163 163 185 185 166 166 151 166 166 186 187 186 187 188 188 150 165 165 165 165 164 164 149 164 164)
            (list 143 143 169 155 155 131 131 143 143 134 134 152 152 162 143 143 143 143 143 143 153 153 165 133 133 167 132 154 154 132)
            (list 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255)
    )
  )

  (defun ALB_m2 ()
    (mapcar 'vector_image
            (list 50 54 53 51 50 48 47 45 44 42 41 39 38 37 35 34 32 32 41 105 107 108 110 111 112 114 115 117 118 120 121 123 124 126 127 123 114 105)
            (list 264 40 47 70 77 99 114 129 144 158 173 188 203 209 232 239 264 264 264 30 47 62 77 91 106 121 136 150 165 180 195 209 224 239 254 264 264 264)
            (list 55 55 53 52 50 49 48 46 45 43 42 40 39 37 36 34 33 37 46 106 107 109 110 112 113 115 116 118 119 121 122 123 125 126 128 128 119 110)
            (list 264 30 55 62 85 91 106 121 136 150 165 180 195 217 224 247 254 264 264 40 55 70 85 99 114 129 144 158 173 188 203 217 232 247 264 264 264 264)
            (list 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2)
    )
    (mapcar 'vector_image
            (list 617 614 612 610 608 607 606 606 606 606 607 608 609 611 613 616 619 511 507 505 502 500 499 498 498 498 498 499 501 503 505 508 512 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 387 391 395 398 400 400 400 400 400 398 392 382 369 355 346 346 348 259 263 266 269 270 269 265 258 249 240 231 223 218 216 216 217)
            (list 42 55 67 80 93 106 119 132 145 158 171 184 197 210 223 236 249 35 49 64 78 93 107 122 137 151 166 181 195 210 224 238 253 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44 262 247 233 220 209 200 195 192 186 173 158 143 126 104 78 55 35 256 242 227 211 196 180 166 152 139 127 114 101 86 71 56 40)
            (list 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 672 564 564 564 564 564 564 564 564 564 564 564 564 564 564 564 564 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 433 439 444 448 452 453 452 448 440 430 419 409 399 393 391 393 396 306 312 317 321 323 318 304 289 277 268 263 261 261 261 263 265)
            (list 42 55 67 80 93 106 119 132 145 158 171 184 197 210 223 236 249 48 61 74 87 100 112 125 137 150 162 175 188 200 213 226 239 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44 279 264 249 234 218 202 186 171 157 145 133 121 108 93 78 62 46 271 255 237 218 196 173 150 133 119 106 93 82 76 69 59 47)
            (list 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1)
    )
    (mapcar 'vector_image
            (list 188 188 188 188 183 180 175 172 175 177 177 177 177 177 176 172 172 169 169 168 167 166 165 164 164 164 161 161 160 158 158 157 156 156 156 156 156 157 158 158 160 161 161 184 187 177 178 179 180 181 182 182 180 178 177 177 177 177 178 179 180 181 172 169 172 174 174 174 174 174 173 169 169 166 159 159 159 619 618 616 615 614 613 612 611 610 609 608 607 606 605 605 605 606 607 608 609 610 611 612 613 614 615 616 617 618 619 620 512 511 510 508 507 506 505 504 503 502 501 500 499 498 497 497 497 498 499 500 501 502 503 504 505 506 507 508 509 510 511 512 672 564 265 264 263 262 261 261 261 262 264 266 268 271 276 280 288 298 304 308 311 313 316 319 321 322 323 322 321 319 318 316 314 310 309 308 305 303 218 217 216 216 216 217 218 219 220 222 224 228 230 232 241 244 250 253 257 260 264 266 268 269 270 269 268 267 265 262 260 257 254 398 396 395 394 393 391 391 391 392 393 394 396 402 404 407 411 418 430 435 439 441 443 446 447 449 451 452 453 452 451 449 447 445 444 441 440 438 435 433 349 348 347 346 345 345 346 347 348 349 350 353 357 363 365 368 371 375 378 383 389 394 395 396 397 398 399 400 399 398 397 395 394 392 391 388 385 384 105 55)
            (list 264 260 256 256 264 256 260 260 260 259 258 258 257 256 256 256 256 256 262 264 264 264 264 263 262 256 263 263 264 264 264 263 262 261 259 258 258 257 256 256 256 256 257 244 244 251 252 252 252 252 250 249 248 247 247 246 245 245 244 244 244 244 248 248 248 247 246 246 245 244 244 244 244 244 248 244 244 252 248 239 235 230 225 219 213 206 199 191 181 167 148 130 130 115 103 93 85 78 71 65 60 55 50 45 41 36 32 28 256 252 248 239 235 230 225 219 213 206 199 191 181 167 148 130 130 115 103 93 85 78 71 65 60 55 50 45 41 36 32 28 42 42 47 52 57 63 73 73 79 89 97 102 106 111 118 123 132 143 150 155 159 162 167 174 180 185 193 215 221 230 234 241 247 259 262 265 273 279 36 44 56 56 77 83 88 91 94 99 103 110 113 116 128 132 140 144 150 155 163 168 175 180 188 212 219 224 233 244 251 261 271 35 43 47 52 57 74 74 80 88 93 97 102 112 115 119 124 132 145 151 156 159 162 167 169 174 180 185 192 214 220 230 237 244 247 256 259 265 273 279 29 36 44 51 63 63 76 83 87 91 94 101 108 117 120 124 128 133 137 144 153 163 165 168 171 175 180 189 212 218 224 233 237 244 248 258 268 271 30 30)
            (list 193 191 193 188 186 183 177 175 177 177 177 177 177 177 177 176 172 169 169 169 168 167 166 165 164 164 162 161 161 160 158 158 157 156 156 156 157 158 158 160 161 161 162 190 187 178 179 180 181 182 182 182 182 180 178 177 177 178 179 180 181 182 174 172 174 174 174 174 174 174 174 173 169 166 162 164 159 620 619 618 616 615 614 613 612 611 610 609 608 607 606 605 606 607 608 609 610 611 612 613 614 615 616 617 618 619 620 622 514 512 511 510 508 507 506 505 504 503 502 501 500 499 498 497 498 499 500 501 502 503 504 505 506 507 508 509 510 511 512 514 672 564 269 265 264 263 262 261 262 264 266 268 271 276 280 288 298 304 308 311 313 316 319 321 322 323 323 323 322 321 319 318 316 314 310 309 308 305 220 218 217 216 217 218 219 220 222 224 228 230 232 241 244 250 253 257 260 264 266 268 269 270 270 270 269 268 267 265 262 260 257 399 398 396 395 394 393 391 392 393 394 396 402 404 407 411 418 430 435 439 441 443 446 447 449 451 452 453 453 453 452 451 449 447 445 444 441 440 438 435 350 349 348 347 346 346 347 348 349 350 353 357 363 365 368 371 375 378 383 389 394 395 396 397 398 399 400 400 400 399 398 397 395 394 392 391 388 385 105 55)
            (list 264 260 256 264 256 264 264 260 259 259 259 258 258 257 256 256 264 262 263 263 264 264 264 264 263 262 262 264 264 264 264 264 263 262 261 259 257 256 256 256 256 257 258 244 252 252 252 252 252 251 251 250 249 248 247 247 246 244 244 244 244 245 252 248 247 247 247 246 246 245 244 244 252 252 248 244 252 256 252 248 239 235 230 225 219 213 206 199 191 181 167 148 115 103 93 85 78 71 65 60 55 50 45 41 36 32 28 21 264 256 252 248 239 235 230 225 219 213 206 199 191 181 167 148 115 103 93 85 78 71 65 60 55 50 45 41 36 32 28 21 262 262 30 47 52 57 63 79 89 97 102 106 111 118 123 132 143 150 155 159 162 167 174 180 185 193 206 206 215 221 230 234 241 247 259 262 265 273 24 36 44 77 83 88 91 94 99 103 110 113 116 128 132 140 144 150 155 163 168 175 180 188 202 202 212 219 224 233 244 251 261 30 35 43 47 52 57 80 88 93 97 102 112 115 119 124 132 145 151 156 159 162 167 169 174 180 185 192 203 203 214 220 230 237 244 247 256 259 265 273 24 29 36 44 51 76 83 87 91 94 101 108 117 120 124 128 133 137 144 153 163 165 168 171 175 180 189 202 202 212 218 224 233 237 244 248 258 268 264 264)
            (list 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250)
    )
    (mapcar 'vector_image
            (list 163 148 148 148 163 163 163 166 166 166 151 151 151 166 166 165 166 165 164 164 150 150 165 165 150 164 164 149 149 149 216 215 214 216 220 227 236 246 255 262 267 269 268 265 261 257 258 259 263 263 266 267 269 269 270 270 269 269 265 265 259 258 250 249 241 239 231 230 223 223 218 218 216 216 216 216 217 218 433 396 393 392 393 398 406 417 427 438 447 453 454 452 448 444 438 422 423 428 428 433 434 437 438 441 441 442 442 442 441 438 438 432 430 422 421 412 410 401 400 391 389 383 382 380 380 382 382 385 385)
            (list 117 131 131 155 155 169 117 162 124 124 134 134 152 152 123 121 163 165 119 167 133 153 153 121 133 154 119 132 154 132 43 51 66 82 97 111 123 136 148 161 176 191 214 229 244 259 257 255 243 241 228 226 212 210 197 195 181 179 167 165 153 151 140 138 128 126 115 113 102 100 87 85 72 70 57 55 41 39 282 49 65 73 88 103 117 129 141 153 166 181 205 221 237 252 267 276 274 261 259 246 245 232 230 217 215 203 201 189 187 176 175 165 163 153 151 141 140 129 127 115 113 97 95 79 77 61 59 45 43)
            (list 163 163 148 163 163 189 189 185 185 166 166 151 166 166 186 187 186 187 188 188 150 165 165 165 165 164 164 149 164 164 217 215 215 217 223 232 241 250 259 265 269 269 269 266 263 259 269 269 273 274 277 277 280 280 281 281 280 280 276 275 268 267 259 258 249 248 241 239 233 232 229 228 227 227 227 227 228 229 436 398 394 392 395 402 411 422 433 443 450 454 455 454 450 446 441 433 433 438 439 443 444 448 449 452 452 453 453 453 452 448 448 441 439 431 429 420 419 409 408 400 399 394 393 391 391 392 393 395 396)
            (list 131 131 155 155 169 143 143 143 143 134 134 152 152 162 143 143 143 143 143 143 153 153 165 133 133 167 132 154 154 132 35 59 74 90 104 117 130 142 155 169 184 199 206 221 236 251 261 259 245 243 230 228 214 212 197 195 180 178 163 161 147 145 133 132 121 119 109 107 97 95 85 83 72 70 57 55 43 41 275 41 57 81 96 110 123 135 147 159 173 189 197 213 229 244 260 280 278 265 263 250 248 235 233 219 217 203 201 187 185 172 170 158 156 146 144 134 132 122 120 109 107 94 92 79 77 63 61 47 45)
            (list 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255)
    )
    (mapcar 'vector_image
            (list 672 672 672 673 673 674 674 674 674 675 675 675 675 676 676 676 676 676 676 676 676 676 675 675 675 675 674 674 674 674 673 673 672 672 671 671 670 670 670 670 669 669 669 669 668 668 668 668 668 668 668 668 668 668 668 669 669 669 669 670 670 670 670 671 671 672 672 673 673 674 674 674 675 675 675 675 675 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 669 669 669 669 669 669 669 670 670 670 671 671 672 672 673 673 673 673 674 674 674 674 674 673 673 673 673 672 672 671 671 671 671 670 670 670 670 670 670 670 671 671 671 671 672 672 673 673 673 672 672 671 671 671 671 671 672 672 673 673 674 674 674 675 675 676 676 676 676 677 677 677 677 677 677 677 677 677 676 676 676 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 668 668 668 667 667 667 667 667 667 667 667 667 667 667 668 668 668 668 669 669 670 670 670 671 671 506 506 506 507 507 508 508 508 508 509 509 509 509 510 510 510 510 510 510 510 510 510 509 509 509 509 508 508 508 508 507 507 506 506 505 505 504 504 504 504 503 503 503 503 502 502 502 502 502 502 502 502 502 502 502 503 503 503 503 504 504 504 504 505 505 506 506 507 507 508 508 508 509 509 509 509 509 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 503 503 503 503 503 503 503 504 504 504 505 505 506 506 507 507 507 507 508 508 508 508 508 507 507 507 507 506 506 505 505 505 505 504 504 504 504 504 504 504 505 505 505 505 506 506 507 507 507 506 506 505 505 505 505 505 506 506 507 507 508 508 508 509 509 510 510 510 510 511 511 511 511 511 511 511 511 511 510 510 510 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 502 502 502 501 501 501 501 501 501 501 501 501 501 501 502 502 502 502 503 503 504 504 504 505 505 441 441 441 442 442 443 443 443 443 444 444 444 444 445 445 445 445 445 445 445 445 445 444 444 444 444 443 443 443 443 442 442 441 441 440 440 439 439 439 439 438 438 438 438 437 437 437 437 437 437 437 437 437 437 437 438 438 438 438 439 439 439 439 440 440 441 441 442 442 443 443 443 444 444 444 444 444 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 438 438 438 438 438 438 438 439 439 439 440 440 441 441 442 442 442 442 443 443 443 443 443 442 442 442 442 441 441 440 440 440 440 439 439 439 439 439 439 439 440 440 440 440 441 441 442 442 442 441 441 440 440 440 440 440 441 441 442 442 443 443 443 444 444 445 445 445 445 446 446 446 446 446 446 446 446 446 445 445 445 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 437 437 437 436 436 436 436 436 436 436 436 436 436 436 437 437 437 437 438 438 439 439 439 440 440 261 261 261 262 262 263 263 263 263 264 264 264 264 265 265 265 265 265 265 265 264 264 264 264 263 263 263 263 262 262 261 261 260 260 259 259 259 259 258 258 258 258 257 257 257 257 257 257 257 257 257 257 257 258 258 258 258 259 259 259 259 260 260 261 261 262 262 263 263 264 264 264 264 264 263 263 262 262 261 261 260 260 259 259 259 258 258 258 258 258 258 258 258 258 259 259 259 260 260 261 261 262 262 262 262 263 263 263 263 263 262 262 262 262 261 261 260 260 259 259 259 259 259 259 259 260 260 261 261 262 262 262 261 261 260 260 260 260 260 260 260 261 261 262 262 263 263 263 264 264 265 265 265 265 266 266 266 266 266 266 266 266 266 265 265 265 265 264 264 264 263 263 263 262 262 261 261 260 260 259 259 259 258 258 257 257 257 256 256 256 256 256 256 256 256 256 256 256 257 257 257 257 258 258 259 259 259 260 260 173 173 173 174 174 175 175 175 175 176 176 176 176 177 177 177 177 177 177 177 177 177 176 176 176 176 175 175 175 175 174 174 173 173 172 172 171 171 171 171 170 170 170 170 169 169 169 169 169 169 169 169 169 169 169 170 170 170 170 171 171 171 171 172 172 173 173 174 174 175 175 175 176 176 176 176 176 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 170 170 170 170 170 170 170 171 171 171 172 172 173 173 174 174 174 174 175 175 175 175 175 174 174 174 174 173 173 172 172 172 172 171 171 171 171 171 171 171 172 172 172 172 173 173 174 174 174 173 173 172 172 172 172 172 173 173 174 174 175 175 175 176 176 177 177 177 177 178 178 178 178 178 178 178 178 178 177 177 177 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 169 169 169 168 168 168 168 168 168 168 168 168 168 168 169 169 169 169 170 170 171 171 171 172 172)
            (list 215 220 220 220 220 220 219 219 219 219 218 218 218 218 217 217 216 216 215 215 214 214 214 214 213 213 213 213 212 212 212 212 212 212 212 212 212 212 212 213 213 213 213 214 214 214 214 215 215 216 216 217 217 218 218 218 218 219 219 219 219 220 220 220 220 219 219 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 213 213 213 213 213 213 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 219 218 218 218 217 217 217 217 216 216 215 215 215 215 214 214 214 214 214 214 214 215 215 215 215 216 216 217 217 217 217 218 218 217 217 216 216 215 215 215 215 215 216 216 217 221 221 221 221 221 220 220 220 220 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 212 212 212 211 211 211 211 211 211 211 211 211 211 211 212 212 213 213 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 220 220 220 220 221 221 221 221 231 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 235 235 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 229 229 229 229 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 235 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 233 233 232 232 231 231 231 231 231 232 232 233 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 227 227 228 228 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 255 260 260 260 260 260 259 259 259 259 258 258 258 258 257 257 256 256 255 255 254 254 254 254 253 253 253 253 252 252 252 252 252 252 252 252 252 252 252 253 253 253 253 254 254 254 254 255 255 256 256 257 257 258 258 258 258 259 259 259 259 260 260 260 260 259 259 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 253 253 253 253 253 253 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 259 258 258 258 257 257 257 257 256 256 255 255 255 255 254 254 254 254 254 254 254 255 255 255 255 256 256 257 257 257 257 258 258 257 257 256 256 255 255 255 255 255 256 256 257 261 261 261 261 261 260 260 260 260 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 252 252 252 251 251 251 251 251 251 251 251 251 251 251 252 252 253 253 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 260 260 260 260 261 261 261 261 247 252 252 252 252 252 251 251 251 251 250 250 250 249 249 248 248 247 247 246 246 246 245 245 245 245 244 244 244 244 244 244 244 244 244 244 244 245 245 245 245 246 246 246 246 247 247 248 248 249 249 250 250 250 250 251 251 251 251 252 252 252 252 251 251 251 251 250 250 249 248 248 247 247 246 246 245 245 245 245 245 245 245 245 246 246 246 247 247 248 248 249 249 250 250 250 251 251 251 250 250 250 249 249 249 249 248 248 247 247 247 247 246 246 246 246 246 246 247 247 247 248 248 249 249 250 250 249 249 248 248 247 247 247 247 247 247 248 248 249 249 253 253 253 253 253 252 252 252 252 251 250 250 250 250 249 249 248 248 247 247 246 246 246 246 245 245 245 244 244 244 243 243 243 243 243 243 243 243 243 243 243 244 244 244 244 245 246 246 246 247 247 248 248 249 249 250 250 251 251 252 252 252 252 252 253 253 253 253 231 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 235 235 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 229 229 229 229 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 235 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 233 233 232 232 231 231 231 231 231 232 232 233 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 227 227 228 228 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237)
            (list 672 672 673 673 674 674 674 674 675 675 675 675 676 676 676 676 676 676 676 676 676 676 676 675 675 675 675 674 674 674 674 673 673 672 672 671 671 670 670 670 670 669 669 669 669 668 668 668 668 668 668 668 668 668 669 669 669 669 670 670 670 670 671 671 672 672 673 673 674 674 674 675 675 675 675 675 675 675 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 669 669 669 669 669 670 670 670 671 671 672 672 673 673 673 673 674 674 674 674 674 674 674 673 673 673 673 672 672 671 671 671 671 670 670 670 670 670 671 671 671 671 672 672 673 673 673 673 673 672 672 671 671 671 672 672 673 673 674 674 674 675 675 676 676 676 676 677 677 677 677 677 677 677 677 677 677 677 676 676 676 675 675 674 674 674 673 673 672 672 671 671 670 670 670 669 669 668 668 668 667 667 667 667 667 667 667 667 667 668 668 668 668 669 669 670 670 670 671 671 672 506 506 507 507 508 508 508 508 509 509 509 509 510 510 510 510 510 510 510 510 510 510 510 509 509 509 509 508 508 508 508 507 507 506 506 505 505 504 504 504 504 503 503 503 503 502 502 502 502 502 502 502 502 502 503 503 503 503 504 504 504 504 505 505 506 506 507 507 508 508 508 509 509 509 509 509 509 509 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 503 503 503 503 503 504 504 504 505 505 506 506 507 507 507 507 508 508 508 508 508 508 508 507 507 507 507 506 506 505 505 505 505 504 504 504 504 504 505 505 505 505 506 506 507 507 507 507 507 506 506 505 505 505 506 506 507 507 508 508 508 509 509 510 510 510 510 511 511 511 511 511 511 511 511 511 511 511 510 510 510 509 509 508 508 508 507 507 506 506 505 505 504 504 504 503 503 502 502 502 501 501 501 501 501 501 501 501 501 502 502 502 502 503 503 504 504 504 505 505 506 441 441 442 442 443 443 443 443 444 444 444 444 445 445 445 445 445 445 445 445 445 445 445 444 444 444 444 443 443 443 443 442 442 441 441 440 440 439 439 439 439 438 438 438 438 437 437 437 437 437 437 437 437 437 438 438 438 438 439 439 439 439 440 440 441 441 442 442 443 443 443 444 444 444 444 444 444 444 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 438 438 438 438 438 439 439 439 440 440 441 441 442 442 442 442 443 443 443 443 443 443 443 442 442 442 442 441 441 440 440 440 440 439 439 439 439 439 440 440 440 440 441 441 442 442 442 442 442 441 441 440 440 440 441 441 442 442 443 443 443 444 444 445 445 445 445 446 446 446 446 446 446 446 446 446 446 446 445 445 445 444 444 443 443 443 442 442 441 441 440 440 439 439 439 438 438 437 437 437 436 436 436 436 436 436 436 436 436 437 437 437 437 438 438 439 439 439 440 440 441 261 261 262 262 263 263 263 263 264 264 264 264 265 265 265 265 265 265 265 265 265 264 264 264 264 263 263 263 263 262 262 261 261 260 260 259 259 259 259 258 258 258 258 257 257 257 257 257 257 257 257 257 258 258 258 258 259 259 259 259 260 260 261 261 262 262 263 263 264 264 264 264 264 264 264 263 263 262 262 261 261 260 260 259 259 259 258 258 258 258 258 258 258 259 259 259 260 260 261 261 262 262 262 262 263 263 263 263 263 263 263 262 262 262 262 261 261 260 260 259 259 259 259 259 260 260 261 261 262 262 262 262 262 261 261 260 260 260 260 260 261 261 262 262 263 263 263 264 264 265 265 265 265 266 266 266 266 266 266 266 266 266 266 266 265 265 265 265 264 264 264 263 263 263 262 262 261 261 260 260 259 259 259 258 258 257 257 257 256 256 256 256 256 256 256 256 256 257 257 257 257 258 258 259 259 259 260 260 261 173 173 174 174 175 175 175 175 176 176 176 176 177 177 177 177 177 177 177 177 177 177 177 176 176 176 176 175 175 175 175 174 174 173 173 172 172 171 171 171 171 170 170 170 170 169 169 169 169 169 169 169 169 169 170 170 170 170 171 171 171 171 172 172 173 173 174 174 175 175 175 176 176 176 176 176 176 176 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 170 170 170 170 170 171 171 171 172 172 173 173 174 174 174 174 175 175 175 175 175 175 175 174 174 174 174 173 173 172 172 172 172 171 171 171 171 171 172 172 172 172 173 173 174 174 174 174 174 173 173 172 172 172 173 173 174 174 175 175 175 176 176 177 177 177 177 178 178 178 178 178 178 178 178 178 178 178 177 177 177 176 176 175 175 175 174 174 173 173 172 172 171 171 171 170 170 169 169 169 168 168 168 168 168 168 168 168 168 169 169 169 169 170 170 171 171 171 172 172 173)
            (list 216 220 220 220 220 220 220 219 219 219 219 218 218 218 218 217 217 216 216 215 215 214 214 214 214 213 213 213 213 212 212 212 212 212 212 212 212 212 213 213 213 213 214 214 214 214 215 215 216 216 217 217 218 218 218 218 219 219 219 219 220 220 220 220 220 219 219 219 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 213 213 213 213 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 219 219 218 218 218 218 217 217 217 217 216 216 215 215 215 215 214 214 214 214 214 215 215 215 215 216 216 217 217 217 217 218 218 218 217 217 217 216 216 215 215 215 216 216 217 217 221 221 221 221 221 221 220 220 220 220 219 219 218 218 218 217 217 216 216 215 215 214 214 214 213 213 212 212 212 211 211 211 211 211 211 211 211 211 212 212 212 212 213 214 214 214 215 215 216 216 217 217 218 218 218 219 219 220 220 220 220 221 221 221 221 221 232 236 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 236 235 235 235 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 229 229 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 235 235 234 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 234 233 233 233 232 232 231 231 231 232 232 233 233 237 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 228 228 228 228 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 237 256 260 260 260 260 260 260 259 259 259 259 258 258 258 258 257 257 256 256 255 255 254 254 254 254 253 253 253 253 252 252 252 252 252 252 252 252 252 253 253 253 253 254 254 254 254 255 255 256 256 257 257 258 258 258 258 259 259 259 259 260 260 260 260 260 259 259 259 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 253 253 253 253 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 259 259 258 258 258 258 257 257 257 257 256 256 255 255 255 255 254 254 254 254 254 255 255 255 255 256 256 257 257 257 257 258 258 258 257 257 257 256 256 255 255 255 256 256 257 257 261 261 261 261 261 261 260 260 260 260 259 259 258 258 258 257 257 256 256 255 255 254 254 254 253 253 252 252 252 251 251 251 251 251 251 251 251 251 252 252 252 252 253 254 254 254 255 255 256 256 257 257 258 258 258 259 259 260 260 260 260 261 261 261 261 261 248 252 252 252 252 252 252 251 251 251 251 250 250 250 249 249 248 248 247 247 246 246 246 245 245 245 245 244 244 244 244 244 244 244 244 244 245 245 245 245 246 246 246 246 247 247 248 248 249 249 250 250 250 250 251 251 251 251 252 252 252 252 252 251 251 251 250 250 249 249 249 248 248 247 247 246 246 245 245 245 245 245 245 246 246 246 247 247 248 248 249 249 250 250 250 251 251 251 251 250 250 250 250 249 249 249 249 248 248 247 247 247 247 246 246 246 246 246 246 247 248 248 249 249 250 250 250 249 249 249 248 248 247 247 247 247 248 248 249 249 249 253 253 253 253 253 253 252 252 251 251 251 250 250 250 250 249 249 248 248 247 247 246 246 246 246 245 245 245 244 244 244 243 243 243 243 243 243 243 243 243 244 244 244 244 245 245 245 246 247 247 248 248 249 249 250 250 251 251 252 252 252 252 252 253 253 253 253 253 232 236 236 236 236 236 236 235 235 235 235 234 234 234 234 233 233 232 232 231 231 230 230 230 230 229 229 229 229 228 228 228 228 228 228 228 228 228 229 229 229 229 230 230 230 230 231 231 232 232 233 233 234 234 234 234 235 235 235 235 236 236 236 236 236 235 235 235 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 229 229 229 229 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 235 235 234 234 234 234 233 233 233 233 232 232 231 231 231 231 230 230 230 230 230 231 231 231 231 232 232 233 233 233 233 234 234 234 233 233 233 232 232 231 231 231 232 232 233 233 237 237 237 237 237 237 236 236 236 236 235 235 234 234 234 233 233 232 232 231 231 230 230 230 229 229 228 228 228 227 227 227 227 227 227 227 227 227 228 228 228 228 229 230 230 230 231 231 232 232 233 233 234 234 234 235 235 236 236 236 236 237 237 237 237 237)
            (list 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10 10)
    )
  )

  (defun ALB_m3 ()
    (mapcar 'vector_image
            (list 50 54 53 51 50 48 47 45 44 42 41 39 38 37 35 34 32 32 41 105 107 108 110 111 112 114 115 117 118 120 121 123 124 126 127 123 114 105)
            (list 264 40 47 70 77 99 114 129 144 158 173 188 203 209 232 239 264 264 264 30 47 62 77 91 106 121 136 150 165 180 195 209 224 239 254 264 264 264)
            (list 55 55 53 52 50 49 48 46 45 43 42 40 39 37 36 34 33 37 46 106 107 109 110 112 113 115 116 118 119 121 122 123 125 126 128 128 119 110)
            (list 264 30 55 62 85 91 106 121 136 150 165 180 195 217 224 247 254 264 264 40 55 70 85 99 114 129 144 158 173 188 203 217 232 247 264 264 264 264)
            (list 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2 2)
    )
    (mapcar 'vector_image
            (list 270 273 276 279 281 281 281 280 277 273 268 263 257 254 252 252 253 429 422 415 409 404 400 397 395 394 394 395 397 400 404 409 416 423 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55 55)
            (list 265 251 238 225 212 200 188 176 164 152 139 125 109 91 74 57 41 279 266 252 238 224 209 194 179 163 148 133 118 103 88 73 59 46 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44)
            (list 323 327 330 333 335 336 336 334 330 325 319 314 310 308 307 307 308 472 465 460 455 452 448 446 445 444 444 445 446 449 452 456 461 466 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105 105)
            (list 279 265 250 234 218 201 183 166 149 134 119 106 94 83 72 59 46 252 242 231 220 209 197 186 174 162 150 138 126 114 103 92 81 70 264 250 236 223 209 195 181 168 154 140 127 113 99 85 71 58 44)
            (list 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1)
    )
    (mapcar 'vector_image
            (list 308 307 307 307 308 309 310 312 315 318 319 321 322 325 327 330 332 333 334 335 336 335 334 333 332 331 330 328 326 323 254 253 252 252 252 253 254 255 257 261 264 267 270 273 275 277 279 280 281 280 279 278 276 275 273 272 271 270 469 466 460 457 454 453 450 448 447 446 445 444 443 443 443 444 445 446 447 448 449 450 451 453 455 457 459 460 464 466 469 425 422 419 415 413 410 408 405 404 402 400 399 398 397 396 395 394 393 393 393 394 395 396 397 399 400 402 405 408 410 413 415 419 420 425 105 55)
            (list 46 61 61 73 84 89 93 100 108 116 119 124 127 135 141 151 159 164 169 175 188 220 228 234 240 245 250 259 267 279 34 44 61 61 73 88 94 100 108 121 129 137 145 153 159 166 174 179 189 217 224 230 239 244 252 256 260 265 64 69 80 87 95 100 107 114 119 124 130 137 150 150 162 175 184 189 194 198 202 206 209 215 221 226 230 232 240 244 249 40 45 51 59 63 70 75 83 86 93 100 104 109 114 119 126 134 148 148 163 178 187 194 200 209 213 220 230 238 243 250 254 262 264 273 30 30)
            (list 309 308 307 308 309 310 312 315 318 319 321 322 325 327 330 332 333 334 335 336 336 336 335 334 333 332 331 330 328 326 255 254 253 252 253 254 255 257 261 264 267 270 273 275 277 279 280 281 281 281 280 279 278 276 275 273 272 271 472 469 466 460 457 454 453 450 448 447 446 445 444 443 444 445 446 447 448 449 450 451 453 455 457 459 460 464 466 469 471 430 425 422 419 415 413 410 408 405 404 402 400 399 398 397 396 395 394 393 394 395 396 397 399 400 402 405 408 410 413 415 419 420 425 429 105 55)
            (list 31 46 73 84 89 93 100 108 116 119 124 127 135 141 151 159 164 169 175 188 203 203 220 228 234 240 245 250 259 267 26 34 44 73 88 94 100 108 121 129 137 145 153 159 166 174 179 189 205 205 217 224 230 239 244 252 256 260 59 64 69 80 87 95 100 107 114 119 124 130 137 162 175 184 189 194 198 202 206 209 215 221 226 230 232 240 244 249 252 32 40 45 51 59 63 70 75 83 86 93 100 104 109 114 119 126 134 163 178 187 194 200 209 213 220 230 238 243 250 254 262 264 273 279 264 264)
            (list 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250 250)
    )
    (mapcar 'vector_image
            (list 448 446 445 441 439 438 435 432 432 431 427 427 426 422 422 423 418 418 421 416 415 419 414 414 419 413 413 419 413 413 419 414 414 421 416 416 424 418 419 427 422 423 431 427 428 436 432 433 442 439 440 163 163 163 148 148 148 163 166 166 166 151 151 151 166 166 165 166 165 164 164 150 150 165 165 150 164 164 149 149 149 296 280 280 279 280 282 287 292 297 302 306 308 309 308 306 303 299 290 291 294 294 297 298 300 300 302 302 303 303 303 302 301 300 298 297 293 293 288 288 283 282 278 278 275 274 274 274 274 275 275)
            (list 261 270 268 249 258 256 237 245 243 224 232 230 211 219 217 198 205 203 185 192 190 171 178 176 158 164 162 144 150 148 139 136 134 126 122 120 112 108 106 99 95 93 86 81 79 74 68 67 61 56 54 117 169 155 155 131 131 117 162 124 124 134 134 152 152 123 121 163 165 119 167 133 153 153 121 133 154 119 132 154 132 275 46 53 68 82 97 111 125 138 152 166 181 195 210 232 246 261 272 270 257 256 243 241 229 227 215 213 201 199 187 185 173 171 160 158 146 144 132 130 119 117 104 102 87 74 72 59 57 44 42)
            (list 453 456 455 445 449 448 439 443 442 434 439 438 429 434 433 425 430 430 422 428 427 420 426 426 419 425 425 419 425 425 420 426 426 423 427 428 426 430 431 430 434 434 435 438 439 440 443 444 447 449 450 189 189 163 163 148 163 163 185 185 166 166 151 166 166 186 187 186 187 188 188 150 165 165 165 165 164 164 149 164 164 298 281 280 280 281 285 289 294 299 304 307 309 309 308 307 304 301 302 302 305 306 309 309 312 312 314 314 315 315 315 314 313 312 309 309 305 304 299 299 294 294 290 289 287 286 286 286 286 287 287)
            (list 269 263 261 257 252 250 245 240 238 233 228 226 220 215 213 207 203 201 194 190 188 180 177 175 167 163 161 153 150 148 130 137 135 117 124 122 104 111 109 91 98 96 78 86 84 66 74 72 54 62 60 143 143 169 155 155 131 131 143 143 134 134 152 152 162 143 143 143 143 143 143 153 153 165 133 133 167 132 154 154 132 267 39 61 75 90 104 118 132 145 159 174 188 203 217 224 239 253 275 273 260 258 246 244 231 229 216 214 201 199 186 184 171 169 156 154 142 140 128 126 114 113 101 99 85 74 72 59 57 45 43)
            (list 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255 255)
    )
  )
  ;;------------------------------------------------------------;;
  ;;              http://www.lee-mac.com/popup.html             ;;  
  ;;-------------------------=={ Popup }==----------------------;;
  ;;                                                            ;;
  ;;  Displays a pop-up message box prompting the user.         ;;
  ;;------------------------------------------------------------;;
  ;;  Author: Lee Mac, Copyright  2011 - www.lee-mac.com       ;;
  ;;------------------------------------------------------------;;
  ;;  Arguments:                                                ;;
  ;;  title - Text to be displayed in the pop-up title bar      ;;
  ;;  msg   - Text content of the pop-up message box            ;;
  ;;  flags - Integer indicating icon & button appearance       ;;
  ;;------------------------------------------------------------;;
  ;;  Returns:  Integer indicating the button pressed to exit   ;;
  ;;------------------------------------------------------------;;
  (defun LM:Popup (title msg flags / wsh res)
    (vl-catch-all-apply
      (function
        (lambda nil
          (setq wsh (vlax-create-object "WScript.Shell"))
          (setq res (vlax-invoke-method wsh 'popup msg 0 title flags))
        )
      )
    )
    (if wsh (vlax-release-object wsh))
    res
  )

  (defun crea_dcl_alb (/ fn f)
    (setq fn DCLname)
    (setq f (open fn "w"))
    (write-line "" f)
    (write-line "ALB : dialog {" f)
    (write-line "label = \"  AlignHM V1.0\";" f)
    (write-line "" f)
    (write-line "    :spacer {}" f)
    (write-line "    " f)
    (write-line "    :image {" f)
    (write-line "    alignment = centered;" f)
    (write-line "    key = \"ALB\";" f)
    (write-line "    width = 90;" f)
    (write-line "    height = 18.5;" f)
    (write-line "    fixed_width = true;" f)
    (write-line "    fixed_height = true;" f)
    (write-line "    aspect_ratio = 1;" f)
    (write-line "    color = dialog_background; " f)
    (write-line "    }" f)
    (write-line "    " f)
    (write-line "    : text { " f)
    (write-line "    label = \"The origin shape should be a rectangle, an isosceles triangle or an isosceles trapezoid\";" f)
    (write-line "    alignment = centered;" f)
    (write-line "    }" f)
    (write-line "" f)
    (write-line "    :spacer {}" f)
    (write-line "    :spacer {}" f)
    (write-line "    :spacer {}" f)
    (write-line "    " f)
    (write-line "    :row {" f)
    (write-line "      " f)
    (write-line "        :boxed_column {" f)
    (write-line "        label = \"Adaptation method\";" f)
    (write-line "" f)
    (write-line "           // :spacer {}" f)
    (write-line "" f)
    (write-line "            :row {" f)
    (write-line "" f)
    (write-line "                :radio_column {" f)
    (write-line "                key = \"align_m\";" f)
    (write-line "" f)
    (write-line "                    :spacer {}" f)
    (write-line "" f)
    (write-line "                    :radio_button {" f)
    (write-line "                    key = \"1\";" f)
    (write-line "                    label = \" 1  -  Subdivision on destination curves\";" f)
    (write-line "                    }" f)
    (write-line "" f)
    (write-line "                    :radio_button {" f)
    (write-line "                    key = \"2\";" f)
    (write-line "                    label = \" 2  -  Perpendicular to the first destination curve\";" f)
    (write-line "                    }" f)
    (write-line "" f)
    (write-line "                    :radio_button {" f)
    (write-line "                    key = \"3\";" f)
    (write-line "                    label = \" 3  -  Perpendicular to the centerline (offset only)\";" f)
    (write-line "                    }" f)
    (write-line "" f)
    (write-line "                    :spacer {}             " f)
    (write-line "" f)
    (write-line "                }" f)
    (write-line "" f)
    (write-line "            }" f)
    (write-line "" f)
    (write-line "            :spacer {}             " f)
    (write-line "             " f)
    (write-line "        }" f)
    (write-line "" f)
    (write-line "        :column {" f)
    (write-line "        " f)
    (write-line "            :boxed_column {" f)
    (write-line "" f)
    (write-line "                :toggle {" f)
    (write-line "                label = \"Delete origin hatch\";" f)
    (write-line "                key = \"del_b\";                " f)
    (write-line "                }" f)
    (write-line "" f)
    (write-line "            }" f)
    (write-line "" f)
    (write-line "            : spacer {}" f)
    (write-line "            " f)
    (write-line "            :row {" f)
    (write-line "" f)
    (write-line "                :spacer {}" f)
    (write-line "            " f)
    (write-line "                : button {" f)
    (write-line "                is_default=true;" f)
    (write-line "                fixed_width=true;" f)
    (write-line "                width=13;" f)
    (write-line "                height=2.5;" f)
    (write-line "                key=\"accept\";" f)
    (write-line "                label= \"OK\";" f)
    (write-line "                }" f)
    (write-line "" f)
    (write-line "                : button {" f)
    (write-line "                fixed_width=true;" f)
    (write-line "                width=13;" f)
    (write-line "                height=2.5;       " f)
    (write-line "                is_cancel=true;" f)
    (write-line "                key=\"cancel\";" f)
    (write-line "                label= \"Cancel\";" f)
    (write-line "                }" f)
    (write-line "" f)
    (write-line "                :spacer {}" f)
    (write-line "" f)
    (write-line "            }" f)
    (write-line "" f)
    (write-line "        }" f)
    (write-line "" f)
    (write-line "    }    " f)
    (write-line "" f)
    (write-line "    : spacer {}" f)
    (write-line "    : spacer {}" f)
    (write-line "    : spacer {}" f)
    (write-line "    : spacer {}" f)
    (write-line "    : spacer {}" f)
    (write-line "" f)
    (write-line "    : text { " f)
    (write-line "    label = \"Copyright    2012  -  Gian Paolo Cattaneo\";" f)
    (write-line "    alignment = centered;" f)
    (write-line "    }" f)
    (write-line "" f)
    (write-line "    : spacer {}" f)
    (write-line "    : spacer {}     " f)
    (write-line "" f)
    (write-line "}                  " f)
    (write-line "" f)
    (write-line "" f)
    (close f)
    (load_dialog fn)
  )

  (defun myerror (s)
    (if (/= s "Function cancelled")
      (princ (strcat "\nError: " s))
    )
    (if DCLname
      (vl-file-delete DCLname)
    )
    (if command-s
      (command-s "_.undo" "_end")
      (vl-cmdf "_.undo" "_end")
    )
    (setvar "cmdecho" cmd)
    (if L1 (redraw L1 4))
    (if L2 (redraw L2 4))
    (if L3 (redraw L3 4))
    (if L4 (redraw L4 4))
    (redraw)
    (prompt "\n") (prompt "\n") (prompt "\n")
    (princ)
  )

  (setq olderr *error* *error* myerror)
  (prompt "\n") (prompt "\n") (prompt "\n")
  (setq cmd (getvar "cmdecho"))
  (setvar "cmdecho" 0)
  (if command-s
    (command-s "_.undo" "_begin")
    (vl-cmdf "_.undo" "_begin")
  )
  (alb_dcl)
  (if (and (esel) (no_segm))
    (repeat n_seg
      (if k (setq k (1+ k)) (setq k 0.0))
      (setq ang (- (/ (+ (angle L1a L1b) (angle L2a L2b)) 2) (/ pi 2.0)))
      (if (not b_lay) (setq b_lay (cdr (assoc 8 (entget block)))) (setq b_lay "0"))
      (if (not (setq b_col (cdr (assoc 62 (entget block))))) (setq b_col 256))
      (if (null RET)
        (progn
          (setq E_L (entlast))
          (if (= del_b 0)
            (progn
              (if command-s
                (command-s "_.copy" block "" "_non" "0,0" "_non" "0,0")
                (vl-cmdf "_.copy" block "" "_non" "0,0" "_non" "0,0")
              )
              (setq block (entlast))
            )
          )
          (if command-s
            (command-s "_.explode" block)
            (vl-cmdf "_.explode" block)
          )
          (setq RET (ssadd))
          (while E_L
            (if (setq E_L (entnext E_L)) (ssadd E_L RET))
          )
          ;;******************* Modified by Gu_xl 2012.10.12*************************************************************
          (vla-GetBoundingBox (vlax-ename->vla-object L1) 'a 'b)
          (vla-GetBoundingBox (vlax-ename->vla-object L2) 'c 'd)
          (setq min_L
                (* 0.01
                   (apply
                     'max
                     (append
                       (apply
                         'mapcar
                         (cons
                           '-
                           (reverse
                             (mapcar 'vlax-safearray->list (list a b))
                           )
                         )
                       )
                             (apply
                               'mapcar
                               (cons
                                 '-
                                 (reverse
                                   (mapcar 'vlax-safearray->list (list c d))
                                 )
                               )
                             )
                     )
                   )
                )
          )
          (setq RETNew (ssadd))
          (repeat (setq cnt (sslength RET))
            (setq en_L (ssname RET (setq cnt (1- cnt))))
            (if (> (setq L_D (distance (setq _p0 (cdr (assoc 10 (setq en_LL (entget en_L))))) (setq _pend (cdr (assoc 11 en_ll))))) min_L)
              (progn
                (setq ang_L (angle _p0 _pend))
                (repeat (fix (/ L_D min_L))
                  (setq _P1 (polar _p0 ang_L min_L))
                  (setq en_LL (subst (cons 10 _P0) (assoc 10 en_LL) en_LL)
                        en_LL (subst (cons 11 _P1) (assoc 11 en_LL) en_LL)
                  )
                  (entmake en_LL)
                  (ssadd (entlast) RETNew)
                  (setq _P0 _P1)
                )
                (setq en_LL (subst (cons 10 _P0) (assoc 10 en_LL) en_LL)
                      en_LL (subst (cons 11 _pend) (assoc 11 en_LL) en_LL)
                )
                (entmod en_LL)
                (ssadd en_L RETNew)
              )
              (ssadd en_L RETNew)
            )
          )
          (setq RET RETNew)
          ;;******************* Modified by Gu_xl 2012.10.12*************************************************************
        )
      )
      (if (and acet-ui-progress-init acet-ui-progress-safe acet-ui-progress-done)
        (setq e_t 1)
      )
      (if (null Lin_B) (LineBl))
      (cond
        ((= align_m 1) (draw_m1))
        ((= align_m 2) (draw_m2))
        ((= align_m 3) (draw_m3))
      )
    )
  )
  (crea_blocco)
  (redraw)
  (if command-s
    (command-s "_.undo" "_end")
    (vl-cmdf "_.undo" "_end")
  )
  (setvar "cmdecho" cmd)
  (prompt "\n") (prompt "\n")
    ;(prompt "\nGeometria Allineata")
  (prompt "\nGeometry Aligned")
  (princ)
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(or (not (vl-catch-all-error-p (vl-catch-all-apply (function vlax-get-acad-object) nil))) (vl-load-com))
;;;(if (or (not (member "geomcal.crx" (arx))) (not (member "geomcal.arx" (arx)))) (arxload "geomcal"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(princ)